<?php
/**
 * This file handles Adyen Config
 *
 * @package WkwcAdyen
 * @version 1.2.0
 */
namespace WkwcAdyen\Helper\Adyen;

defined( 'ABSPATH' ) || exit; // Exit if access directly.

/**
 * Adyen Config class
 */
class Wkwc_Adyen_Config {

	/**
	 * $payment_mode
	 *
	 * @var string
	 */
	private $payment_mode;

	/**
	 * $environment
	 *
	 * @var string
	 */
	private $environment;

	/**
	 * $api_key
	 *
	 * @var string
	 */
	private $api_key;

	/**
	 * $merchant_code
	 *
	 * @var string
	 */
	private $merchant_code;

	/**
	 * $origin_key
	 *
	 * @var string
	 */
	private $origin_key;

	/**
	 * $capture_immediately
	 *
	 * @var boolean
	 */
	private $capture_immediately;

	/**
	 * $hmac_key
	 *
	 * @var string
	 */
	private $hmac_key;

	/**
	 * $refund_in_adyen
	 *
	 * @var boolean
	 */
	private $refund_in_adyen;

	/**
	 * $live_endpoint_prefix
	 *
	 * @var string
	 */
	private $live_endpoint_prefix;

	/**
	 * Adyen_Config constructor.
	 *
	 * @param array $settings options data.
	 */
	public function __construct( $settings ) {
		if ( isset( $settings['payment_mode'] ) ) {
			$this->wkwc_adyen_set_payment_mode( $settings['payment_mode'] );
		}

		$this->wkwc_adyen_set_environment( $settings['test_mode'] );
		$environment_prefix = ( 'yes' === $settings['test_mode'] ) ? 'test_' : '';
		$this->wkwc_adyen_set_apikey( $settings[ $environment_prefix . 'api_key' ] );
		$this->wkwc_adyen_set_merchantcode( $settings[ $environment_prefix . 'merchant' ] );
		$this->wkwc_adyen_set_origin_key( $settings[ $environment_prefix . 'origin_key' ] );
		$this->wkwc_adyen_set_capture_immediately( 'yes' === $settings['capture_immediately'] );
		$this->wkwc_adyen_set_hmackey( $settings['hmac_key'] );
		$this->wkwc_adyen_set_refund_in_adyen( 'yes' === $settings['refund_in_adyen'] );
		$this->wkwc_adyen_set_live_endpoint_prefix( $settings['live_endpoint_prefix'] );
	}

	/**
	 * Get Payment Mode
	 *
	 * @return string
	 */
	public function wkwc_adyen_get_payment_mode() {
		return $this->payment_mode;
	}

	/**
	 * Set Payment Mode
	 *
	 * @param string $payment_mode payment mode.
	 * @return string
	 */
	public function wkwc_adyen_set_payment_mode( $payment_mode ) {
		$this->payment_mode = trim( $payment_mode );
		return $this;
	}

	/**
	 * Get Payment Environment
	 *
	 * @return string
	 */
	public function wkwc_adyen_get_environment() {
		return $this->environment;
	}

	/**
	 * Set Payment Environment
	 *
	 * @param string $test_mode payment mode.
	 *
	 * @return Wkwc_Adyen_Config
	 */
	public function wkwc_adyen_set_environment( $test_mode ) {
		$this->environment = ( 'yes' === $test_mode ) ? \Adyen\Environment::TEST : \Adyen\Environment::LIVE;
		return $this;
	}

	/**
	 * Get Adyen Account Api Key
	 *
	 * @return string
	 */
	public function wkwc_adyen_get_apikey() {
		return html_entity_decode( $this->api_key );
	}

	/**
	 * Set Adyen Account Api Key
	 *
	 * @param string $api_key Api Key.
	 *
	 * @return Wkwc_Adyen_Config
	 */
	public function wkwc_adyen_set_apikey( $api_key ) {
		$this->api_key = trim( $api_key );

		return $this;
	}

	/**
	 * Get Adyen Account Merchant Code
	 *
	 * @return string
	 */
	public function wkwc_adyen_get_merchantcode() {
		return $this->merchant_code;
	}

	/**
	 * Set Adyen Account Merchant Code
	 *
	 * @param string $merchant_code Merchant Code.
	 *
	 * @return Wkwc_Adyen_Config
	 */
	public function wkwc_adyen_set_merchantcode( $merchant_code ) {
		$this->merchant_code = trim( $merchant_code );

		return $this;
	}

	/**
	 * Get Adyen Account Origin Key
	 *
	 * @return string
	 */
	public function wkwc_adyen_get_origin_key() {
		return $this->origin_key;
	}

	/**
	 * Set Adyen Account Origin Key
	 *
	 * @param string $origin_key Origin Key.
	 *
	 * @return Wkwc_Adyen_Config
	 */
	public function wkwc_adyen_set_origin_key( $origin_key ) {
		$this->origin_key = trim( $origin_key );

		return $this;
	}

	/**
	 * Get Check Capture availability
	 *
	 * @return boolean
	 */
	public function wkwc_adyen_is_capture_immediately() {
		return $this->capture_immediately;
	}

	/**
	 * Set Check Capture availability
	 *
	 * @param bool $capture_immediately .
	 *
	 * @return Wkwc_Adyen_Config
	 */
	public function wkwc_adyen_set_capture_immediately( $capture_immediately ) {
		$this->capture_immediately = $capture_immediately;

		return $this;
	}

	/**
	 * Get Hmac Key
	 *
	 * @return string
	 */
	public function wkwc_adyen_get_hmackey() {
		return $this->hmac_key;
	}

	/**
	 * Set Hmac Key
	 *
	 * @param string $hmac_key Hmac Key.
	 *
	 * @return Wkwc_Adyen_Config
	 */
	public function wkwc_adyen_set_hmackey( $hmac_key ) {
		$this->hmac_key = trim( $hmac_key );

		return $this;
	}

	/**
	 * Get Check Refund availability
	 *
	 * @return boolean
	 */
	public function wkwc_adyen_is_refund_in_adyen() {
		return $this->refund_in_adyen;
	}

	/**
	 * Set Check Refund availability
	 *
	 * @param bool $refund_in_adyen .
	 *
	 * @return Wkwc_Adyen_Config
	 */
	public function wkwc_adyen_set_refund_in_adyen( $refund_in_adyen ) {
		$this->refund_in_adyen = $refund_in_adyen;

		return $this;
	}

	/**
	 * Get Live Endpoint Prefix
	 *
	 * @return boolean
	 */
	public function wkwc_adyen_get_live_endpoint_prefix() {
		return $this->live_endpoint_prefix;
	}

	/**
	 * Set Live Endpoint Prefix
	 *
	 * @param mixed $live_endpoint_prefix .
	 */
	public function wkwc_adyen_set_live_endpoint_prefix( $live_endpoint_prefix ) {
		$this->live_endpoint_prefix = $live_endpoint_prefix;
	}

	/**
	 * Checks whether or not is the test mode active.
	 *
	 * @return bool
	 */
	public function wkwc_adyen_is_test_mode() {
		return ! empty( $this->wkwc_adyen_get_environment() ) && 'test' === $this->wkwc_adyen_get_environment() ? true : false;
	}

	/**
	 * The authorized flag based on the environment.
	 *
	 * @return string
	 */
	public function wkwc_adyen_env_flag() {
		return empty( $this->wkwc_adyen_get_environment() ) ? 'is_authorized' : "is_authorized_{$this->wkwc_adyen_get_environment()}";
	}
}
