<?php
/**
 * Api Handler
 */

namespace WkwcAdyen\Helper\Common;

use WkwcAdyen\Helper\Adyen\Wkwc_Adyen_Helper;

class Wkwc_Adyen_Api_Handler {

	/**
	 * Base Url
	 *
	 * @var $base_url
	 */
	protected $base_url;

	/**
	 * $adyen_helper
	 *
	 * @var Wkwc_Adyen_Helper
	 */
	protected $adyen_helper;

	/**
	 * $adyen_config
	 *
	 * @var Wkwc_Adyen_Config
	 */
	protected $adyen_config;

	/**
	 * Request arguments.
	 *
	 * @var array
	 */
	protected $args = array();

	// Constructor to set up base URL
	public function __construct( $base_url ) {
		$this->base_url     = untrailingslashit( $base_url );
		$this->adyen_helper = new Wkwc_Adyen_Helper();
		$this->adyen_config = $this->adyen_helper->adyen_config;
	}

	/**
	 * List of headers.
	 *
	 * @return array
	 */
	public function wkwc_adyen_remote_headers() {
		$merchant_name = $this->adyen_config->wkwc_adyen_get_merchantcode();
		$get_api_key   = $this->adyen_config->wkwc_adyen_get_apikey();

		return array(
			'Content-Type'               => 'application/json',
			'X-API-Key'                  => $get_api_key,
			'x-wkwc-domain'              => parse_url( home_url(), PHP_URL_HOST ),
			'x-wkwc-plugin-slug'         => plugin_basename( WKADYEN_FILE ),
			'x-wkwc-plugin-version'      => WKADYEN_VERSION,
			'x-wkwc-adyen-merchant-name' => $merchant_name,
		);
	}

	// Generic request function using wp_remote_request
	public function wkwc_adyen_remote_request( $method, $endpoint, $data = array(), $params = array() ) {
		$url = add_query_arg( $params, $this->base_url . '/' . $endpoint );

		// Set up arguments for the request
		$args = array(
			'method'     => strtoupper( $method ),
			'headers'    => $this->wkwc_adyen_remote_headers(),
			'authorized' => $this->adyen_helper->wkwc_adyen_is_configured(),
		);

		if ( ! empty( $data ) ) {
			$args['body'] = wp_json_encode( $data );
		}

		// Execute the request
		$response = wp_remote_request( $url, $args );

		if ( is_wp_error( $response ) ) {
			$code = $response->get_error_code();
			$body = array(
				'service' => 'wp_plugin',
				'message' => $response->get_error_message(),
			);

			$result = (object) array(
				'status' => 417,
				'code'   => $code,
				'url'    => $url,
				'args'   => $args,
				'body'   => $body,
			);

		} else {
			$headers = wp_remote_retrieve_headers( $response )->getAll();
			$code    = wp_remote_retrieve_response_code( $response );
			$body    = json_decode( wp_remote_retrieve_body( $response ), true );

			$result = (object) array(
				'headers' => $headers,
				'status'  => $code,
				'url'     => $url,
				'args'    => $args,
				'body'    => $body,
			);
		}

		$this->wkwc_adyen_maybe_set_logs( $result );

		return $result;
	}

	/**
	 * Set Api Log
	 *
	 * @param  object $result
	 *
	 * @return array
	 */
	public function wkwc_adyen_maybe_set_logs( $result ) {
		$data = array(
			'title'   => '==== WKWC ADYEN REMOTE REQUEST ====',
			'message' => 'This is a performed remote request.',
			'data'    => array(
				'request'  => array_merge(
					array(
						'endpoint' => $result->url,
					),
					$result->args
				),
				'response' => $result->body,
			),
		);

		wkwc_adn_log( 'Remote request response data: ' . print_r( $data, true ) );
	}



	// GET request
	public function wkwc_adyen_remote_get( $endpoint, $params = array() ) {
		return $this->wkwc_adyen_remote_request( 'GET', $endpoint, array(), $params );
	}

	// POST request
	public function wkwc_adyen_remote_post( $endpoint, $data = array() ) {
		return $this->wkwc_adyen_remote_request( 'POST', $endpoint, $data );
	}

	// PUT request
	public function wkwc_adyen_remote_put( $endpoint, $data = array() ) {
		return $this->wkwc_adyen_remote_request( 'PUT', $endpoint, $data );
	}

	// DELETE request
	public function wkwc_adyen_remote_delete( $endpoint, $data = array() ) {
		return $this->wkwc_adyen_remote_request( 'DELETE', $endpoint, $data );
	}


}
