<?php
/**
 * Credit Card
 */

namespace WkwcAdyen\Includes\Payment_Methods;

// prevent direct access data leaks.
use Automattic\WooCommerce\StoreApi\Utilities\NoticeHandler;

defined( 'ABSPATH' ) || exit;

use WkwcAdyen\Helper\Common\Wkwc_Adyen_Checkout_Action;

use WkwcAdyen\Helper\Common\Wkwc_Adyen_Service_Util;

class Wkwc_Adyen_Credit_Card extends Wkwc_Adyen_Abstract_Gateway {

	/**
	 * Checkout action instance
	 *
	 * @var Wkwc_Adyen_Checkout_Action
	 */
	public $checkout_action;

	/**
	 * Mode Of Payment
	 *
	 * @var $payment_mode
	 */
	public $payment_mode;

	/**
	 * Constructor of this class.
	 *
	 * @param bool $init_hooks
	 * @since 1.0.0
	 */
	public function __construct( $init_hooks = true ) {

		parent::__construct( $init_hooks );

		$this->has_fields      = true;
		$this->checkout_action = new Wkwc_Adyen_Checkout_Action();
		$this->payment_mode    = $this->get_option( 'payment_mode' );
	}

	/**
	 * Gets default payment method title.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function get_default_title() {
		return __( 'Adyen - Credit Card', 'wkwc-adyen' );
	}

	/**
	 * Gets default payment method description.
	 *
	 * @since 1.1.0 - display supported countries
	 * @since 1.0.0
	 * @return string
	 */
	public function get_default_description() {
		return $this->show_supported_country();
	}

	/**
	 * Gets default description set in settings.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function get_settings_description() {
		$this->description = ( $this->settings['description'] ) ? $this->settings['description'] : __( 'Pay with credit card', 'wkwc-adyen' );
		return $this->description;
	}

	/**
	 * Type of the payment method (e.g ideal, scheme. bcmc).
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function payment_method_type() {
		return 'scheme';
	}

	/**
	 * Returns the payment method to be used for recurring payments
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function recurring_payment_method() {
		return $this->payment_method_type();
	}

	/**
	 * Add payment fields
	 *
	 * @return void
	 */
	public function payment_fields() {
		$amount       = WC()->cart->get_total( '' );
		$payment_mode = $this->get_option( 'payment_mode' );

		if ( 'payment_links' === $payment_mode ) {
			if ( $this->get_option( 'payment_link_desc' ) ) {
				echo esc_html( $this->get_option( 'payment_link_desc' ) );
			} else {
				echo esc_html( $this->get_default_description() );
			}
			return;
		}

		$payment_methods['paymentMethods'] = $this->checkout_action->wkwc_adyen_get_payment_methods( null, $amount );

		echo $this->generate_extra_fields_html();
	}

	/**
	 * Generates extra fields HTML.
	 *
	 * @return string
	 */
	public function generate_extra_fields_html() {

		$type         = $this->payment_method_type();
		$installments = ( $this->get_option( 'allow_installments' ) === 'yes' ) ? Wkwc_Adyen_Service_Util::get_installments_by_country( WC()->customer->get_shipping_country(), $this->get_option( 'installments' ) ) : '';

		?>
		<div class="wkwc-adyen-wrap-form">

		<?php $this->render_card_form(); ?>

		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-number" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-card-number" ); ?>" />
		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-exp-month" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-card-exp-month" ); ?>" />
		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-exp-year" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-card-exp-year" ); ?>" />
		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-cvc" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-card-cvc" ); ?>" />

		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-holder" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-card-holder" ); ?>" />
		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-sci" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-sci" ); ?>" />
		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-store-card" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-store-card" ); ?>" />
		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-is-stored-card" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-is-stored-card" ); ?>" value="yes" />
		<input type="hidden" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-installments" ); ?>" name="<?php echo esc_attr( "wkwc-adyen-{$type}-card-installments" ); ?>" />

		<input type="hidden" data-wkwc-adyen-card-installments value='<?php echo esc_attr( json_encode( $installments ) ); ?>' />

		</div>
		<?php
	}

	/**
	 * Renders the form for card fields.
	 *
	 * @since 1.1.1 - improve custom attributes and made the logic more clean
	 * @since 1.0.6 - remove duplicated card-form element
	 * @since 1.0.3
	 * @return string
	 */
	public function render_card_form() {
		$type  = $this->payment_method_type();
		$cards = $this->checkout_action->get_ec_stored_cards();
		?>

		<div class="wkwc-adyen-stored-cards">

		<?php
		if ( ! empty( $cards ) && is_array( $cards ) ) {
			foreach ( $cards as $index => $item ) :

				// exclude BCMC from Credit Cards and Credit Cards from BCMC
				if ( ( 'bcmc' === $this->payment_method_type() && 'bcmc' !== $item['brand'] ) ||
				( 'scheme' === $this->payment_method_type() && 'bcmc' === $item['brand'] ) ) {
					continue;
				}
				?>

			<div class="wkwc-adyen-stored-card is-stored-card">
				<div class="wkwc-adyen-stored-card__details" data-wkwc-adyen-stored-card="<?php echo esc_attr( "wkwc-adyen-{$type}-card-{$index}" ); ?>" data-wkwc-adyen-stored-card-type="<?php echo esc_attr( $type ); ?>">
					<img src="<?php echo esc_attr( 'https://checkoutshopper-test.adyen.com/checkoutshopper/images/logos/' . $item['brand'] ); ?>.svg" alt="">
					<div>******<?php echo esc_html( $item['lastFour'] ); ?></div>
				</div>
				<div class="wkwc-adyen-stored-card__fields" style="display: none;" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-{$index}" ); ?>"></div>
			</div>
					<?php
			endforeach;
		}
		?>

		<div class="wkwc-adyen-stored-card">
			<div class="wkwc-adyen-stored-card__details" data-wkwc-adyen-stored-card="<?php echo esc_attr( "wkwc-adyen-{$type}-card-new" ); ?>" data-wkwc-adyen-stored-card-type="<?php echo esc_attr( $type ); ?>">
				<span class="dashicons dashicons-plus"></span>
				<div><?php _e( 'Use a new card', 'wkwc-adyen' ); ?></div>
			</div>
			<div class="wkwc-adyen-stored-card__fields" id="<?php echo esc_attr( "wkwc-adyen-{$type}-card-new" ); ?>" style="display: none;">
				<div id="wkwc-adyen-card-form"></div>
			</div>
		</div>

		</div>

		<?php
	}

	/**
	 * Get the card installments
	 *
	 * @return string|null
	 */
	public function get_installments() {
		$installments = array();

		if ( is_checkout() && ! empty( WC()->customer ) ) {

			$installments = ( $this->get_option( 'allow_installments' ) === 'yes' ) ? Wkwc_Adyen_Service_Util::get_installments_by_country( WC()->customer->get_shipping_country(), $this->get_option( 'installments' ) ) : '';

		}

		return json_encode( $installments );
	}

	/**
	 * Get the stored cards
	 *
	 * @return array
	 */
	public function get_stored_cards() {

		$cards = $this->checkout_action->get_ec_stored_cards();

		$stored_cards = array();

		foreach ( $cards as $index => $item ) {

			if ( ( 'bcmc' === $this->payment_method_type() && 'bcmc' !== $item['brand'] ) ||
			( 'scheme' === $this->payment_method_type() && 'bcmc' === $item['brand'] ) ) {
				continue;
			}

			$stored_cards[] = $item;
		}

		return $stored_cards;
	}

	/**
	 * Validates extra added fields.
	 *
	 * @since 1.0.0
	 * @return bool
	 */
	public function validate_fields() {

		$payment_mode = $this->get_option( 'payment_mode' );

		$card_holder_is_required = $this->get_option( 'require_card_holder_name' );

		if ( 'payment_links' === $payment_mode ) {
			return true;
		}

		$is_stored_card = isset( $_POST['wkwc-adyen-scheme-is-stored-card'] ) && 'yes' === $_POST['wkwc-adyen-scheme-is-stored-card'] ? 'yes' : 'no';

		$adyen_payment_data = $this->wkwc_adyen_get_adyen_payment_data();

		$payment_method = ( is_null( $adyen_payment_data ) ) ? array() : $adyen_payment_data['paymentMethod'];

		if ( 'yes' === $is_stored_card ) {
			$stored_card_id = ( isset( $_POST['wkwc-adyen-scheme-sci'] ) && ! empty( $_POST['wkwc-adyen-scheme-sci'] ) ) ? $_POST['wkwc-adyen-scheme-sci'] : '';

			if ( empty( $stored_card_id ) ) {
				wc_add_notice( __( 'Please provide the CVC/CVV of the card.', 'wkwc-adyen' ), 'error' );
				return false;
			}
		} else {
			if ( 'yes' === $card_holder_is_required && ( ! isset( $payment_method['holderName'] ) || empty( $payment_method['holderName'] ) ) ) {
				wc_add_notice( esc_html__( 'Card holder name is required!', 'wkwc-adyen' ), 'error' );
				return false;
			}

			if ( ! isset( $payment_method['encryptedCardNumber'] ) || empty( $payment_method['encryptedCardNumber'] ) ) {
				wc_add_notice( esc_html__( 'Credit card number is required!', 'wkwc-adyen' ), 'error' );
				return false;
			}

			if ( ! isset( $payment_method['encryptedExpiryMonth'] ) || empty( $payment_method['encryptedExpiryMonth'] ) ) {
				wc_add_notice( esc_html__( 'Card expiry detail is required!', 'wkwc-adyen' ), 'error' );
				return false;
			}

			if ( ! isset( $payment_method['encryptedSecurityCode'] ) || empty( $payment_method['encryptedSecurityCode'] ) ) {
				wc_add_notice( esc_html__( 'Card verification code is required!', 'wkwc-adyen' ), 'error' );
				return false;
			}

			if ( 'yes' === $card_holder_is_required && ! empty( $payment_method['holderName'] ) && ! preg_match( '/^[a-zA-Z\s]+$/', $payment_method['holderName'] ) ) {
				wc_add_notice( esc_html__( 'Only alphabets and whitespace are allowed.', 'wkwc-adyen' ), 'error' );
				return false;
			}
		}

		return true;
	}

	/**
	 * Adyen Payment Data
	 *
	 * @return array
	 */
	private function wkwc_adyen_get_adyen_payment_data() {
		if ( isset( $_POST['wkwc-adyen-scheme-card-number'] ) ) {

			$card_number = ( isset( $_POST['wkwc-adyen-scheme-card-number'] ) && ! empty( $_POST['wkwc-adyen-scheme-card-number'] ) ) ? $_POST['wkwc-adyen-scheme-card-number'] : '';

			$card_exp_month = ( isset( $_POST['wkwc-adyen-scheme-card-exp-month'] ) && ! empty( $_POST['wkwc-adyen-scheme-card-exp-month'] ) ) ? $_POST['wkwc-adyen-scheme-card-exp-month'] : '';

			$card_exp_year = ( isset( $_POST['wkwc-adyen-scheme-card-exp-year'] ) && ! empty( $_POST['wkwc-adyen-scheme-card-exp-year'] ) ) ? $_POST['wkwc-adyen-scheme-card-exp-year'] : '';

			$card_csv = ( isset( $_POST['wkwc-adyen-scheme-card-cvc'] ) && ! empty( $_POST['wkwc-adyen-scheme-card-cvc'] ) ) ? $_POST['wkwc-adyen-scheme-card-cvc'] : '';

			$card_holder = ( isset( $_POST['wkwc-adyen-scheme-card-holder'] ) && ! empty( $_POST['wkwc-adyen-scheme-card-holder'] ) ) ? $_POST['wkwc-adyen-scheme-card-holder'] : '';

			$browser_info = ( isset( $_POST['wkwc-adyen-scheme-browser-info'] ) && ! empty( $_POST['wkwc-adyen-scheme-browser-info'] ) ) ? json_decode( stripcslashes( $_POST['wkwc-adyen-scheme-browser-info'] ), true ) : '';

			$payment_data = array(
				'paymentMethod' => array(
					'type'                  => 'scheme',
					'encryptedCardNumber'   => $card_number,
					'encryptedExpiryMonth'  => $card_exp_month,
					'encryptedExpiryYear'   => $card_exp_year,
					'holderName'            => $card_holder,
					'encryptedSecurityCode' => $card_csv,
				),
				'browserInfo'   => $browser_info,
			);

		} else {
			$request_data = isset( $_REQUEST ) ? wc_clean( $_REQUEST ) : array();

			if ( ! isset( $request_data['adyen_payment_data'] ) ) {
				return array();
			}

			$payment_data = json_decode( stripcslashes( $request_data['adyen_payment_data'] ), true );
		}

		return $this->wkwc_adyen_recursive_sanitize_text_field( $payment_data );
	}

	/**
	 * Recursive sanitation for an array
	 *
	 * @param array $array sanitize data.
	 *
	 * @return mixed
	 */
	private function wkwc_adyen_recursive_sanitize_text_field( $array ) {
		if ( ! empty( $array ) ) {
			foreach ( $array as &$value ) {
				if ( is_array( $value ) ) {
					$value = $this->wkwc_adyen_recursive_sanitize_text_field( $value );
				} else {
					$value = sanitize_text_field( $value );
				}
			}
		}

		return $array;
	}

	/**
	 * Builds the required payment payload
	 *
	 * @param \WC_Order $order
	 * @param string    $reference
	 * @return array
	 */
	public function build_payment_payload( \WC_Order $order, $reference ) {

		$type = $this->payment_method_type();

		$card_number    = $_POST[ "wkwc-adyen-{$type}-card-number" ];
		$card_exp_month = $_POST[ "wkwc-adyen-{$type}-card-exp-month" ];
		$card_exp_year  = $_POST[ "wkwc-adyen-{$type}-card-exp-year" ];
		$card_cvc       = $_POST[ "wkwc-adyen-{$type}-card-cvc" ];
		$card_holder    = $_POST[ "wkwc-adyen-{$type}-card-holder" ];

		$is_stored_card = $_POST[ "wkwc-adyen-{$type}-is-stored-card" ] ?? 'no';
		$stored_card_id = $_POST[ "wkwc-adyen-{$type}-sci" ];

		$installments = (int) $_POST[ "wkwc-adyen-{$type}-card-installments" ];
		$store_card   = (bool) $_POST[ "wkwc-adyen-{$type}-store-card" ];
		$store_card   = is_user_logged_in() ? $store_card : false;

		$payload = array_merge(
			parent::build_payment_payload( $order, $reference ),
			array(
				'additionalData'     => array(
					'allow3DS2' => false,
				),
				'storePaymentMethod' => $store_card,
			)
		);

		if ( 'yes' === $is_stored_card ) {
			$payload['paymentMethod']['storedPaymentMethodId'] = $stored_card_id;
		} else {
			$payload['paymentMethod']['encryptedCardNumber']  = $card_number;
			$payload['paymentMethod']['encryptedExpiryMonth'] = $card_exp_month;
			$payload['paymentMethod']['encryptedExpiryYear']  = $card_exp_year;
			$payload['paymentMethod']['holderName']           = $card_holder;
		}

		if ( ! empty( $card_cvc ) ) {
			$payload['paymentMethod']['encryptedSecurityCode'] = $card_cvc;
		}

		if ( $installments > 0 ) {
			$payload['installments']['value'] = $installments;
		}

		return $payload;
	}


	/**
	 * Processes the payment.
	 *
	 * @param int $order_id
	 * @return array
	 */
	public function process_payment( $order_id ) {
		$order        = wc_get_order( $order_id );
		$payment_mode = $this->get_option( 'payment_mode' );
		$user_id      = get_current_user_id();

		if ( ! $user_id ) {
			$session = new \WC_Session_Handler();
			$user_id = $session->generate_customer_id();

			$order->update_meta_data( 'wkadyen_order_guest_id', $user_id );
		}

		if ( 'payment_links' === $payment_mode ) {
			$order->update_meta_data( 'wkadyen_mode', 'payment_links' );
			$redirect_url = $this->get_return_url( $order );
			$result       = $this->adyen_helper->wkwc_adyen_pay_via_link( $order, $redirect_url );

			if ( ! empty( $result['url'] ) ) {
				// Payment went through.
				$order->update_meta_data( '_woocommerce_' . $this->id . '_payment_data', $result );
				$order->save();

				return array(
					'result'   => 'success',
					'redirect' => $result['url'],
				);
			} elseif ( ! empty( $result['message'] ) ) {
				$order->add_order_note( $result['message'] );
				wc_add_notice( $result['message'], 'error' );
			} else {
				$order->add_order_note( esc_html__( 'Opps! something went wrong in payment authorize.', 'wkwc-adyen' ) );
				wc_add_notice( esc_html__( 'Opps! something went wrong in payment authorize.', 'wkwc-adyen' ), 'error' );
			}

			return null;
		}

		parent::process_payment( $order_id );

		$payload = $this->build_payment_payload( $order, $order_id );

		$response = $this->checkout_action->send_payment( $payload );

		if ( $response->status == 200 ) {

			return $this->process_payment_result( $response, $order );

		} else {

			wc_add_notice( $response->body['message'], 'error' );

			NoticeHandler::convert_notices_to_exceptions( 'woocommerce_rest_payment_error' );

		}

		return array( 'result' => 'failure' );
	}

	/**
	 * Processes the payment result.
	 *
	 * @since 1.0.6 - use default checkout url if there are subscriptions
	 * @since 1.0.4 - combine all actions in one popup
	 * @since 1.0.0
	 * @param object    $response
	 * @param \WC_Order $order
	 * @return array
	 */
	protected function process_payment_result( $response, $order ) {
		$body_response = $this->adyen_helper::wkwc_adyen_obj_to_arr( $response->body );
		$order_id      = $order->get_id();

		wkwc_adn_log( "Adyen Authorize payment response for order id: $order_id: " . print_r( $body_response, true ) );

		$result_code = isset( $body_response['resultCode'] ) ? $body_response['resultCode'] : '';
		$reference   = isset( $body_response['pspReference'] ) ? $body_response['pspReference'] : '';
		$action      = isset( $body_response['action'] ) ? $body_response['action'] : array();

		$extra_action_codes = array( 'RedirectShopper', 'ChallengeShopper', 'IdentifyShopper' );
		$error_codes        = array( 'Refused', 'Error', 'Cancelled' );

		$result = array(
			'result'   => 'success',
			'redirect' => Wkwc_Adyen_Service_Util::get_return_page_url( $order, $result_code ),
		);

		$order->read_meta_data();

		$order_psp_reference = $order->get_meta( '_' . WKWC_ADN_PREFIX . '_payment_pspReference' );

		if ( $order_psp_reference !== $reference ) {

			$order->update_meta_data( '_' . WKWC_ADN_PREFIX . '_payment_pspReference', $reference );

		}

		$order->update_meta_data( '_' . WKWC_ADN_PREFIX . '_payment_resultCode', $result_code );

		$order->update_meta_data( '_' . WKWC_ADN_PREFIX . '_payment_action', $action );

		$order->save();

		wkwc_adn_log( "Result code: $result_code" );

		if ( in_array( $result_code, $extra_action_codes ) ) {

			$checkout_url = self::has_subscription() ? wc_get_checkout_url() : $order->get_checkout_payment_url();

			if ( 'RedirectShopper' === $result_code ) {
				$result_action = empty( $action ) ? array() : $action;
				$result_type   = empty( $result_action['type'] ) ? '' : $result_action['type'];

				wkwc_adn_log( "Result type: $result_type" );

				if ( 'redirect' === $result_type ) {
					update_option( 'wkwc_adyen_redirect_object_' . $order_id, $action );

					return array(
						'result'   => 'success',
						'redirect' => $order->get_checkout_payment_url( true ),
					);
				}
			}
			// redirect to process payment action via Web Component
			$result = array(
				'result'   => 'success',
				'redirect' => add_query_arg(
					array(
						WKWC_ADN_PREFIX . '_payment_method' => $this->payment_method_type(),
						WKWC_ADN_PREFIX . '_order_id' => $order_id,
					),
					$checkout_url
				),
			);

		} elseif ( in_array( $result_code, $error_codes ) ) {

			wc_add_notice( __( 'The transaction could not be processed! Either it was refused or an error has occurred! Please make sure you provide the valid card details or try with a different card.', 'wkwc-adyen' ), 'error' );

			NoticeHandler::convert_notices_to_exceptions( 'woocommerce_rest_payment_error' );

			$result = array(
				'result' => 'failure',
			);

		} else {
			// Payment went through.
			$order->update_meta_data( '_woocommerce_' . $this->id . '_payment_data', $body_response );

			if ( $this->adyen_config->wkwc_adyen_is_capture_immediately() ) {
				$order->update_meta_data( '_wkwc_adyen_payment_captured', 'yes' );
				$order->set_transaction_id( $reference );
				$order->payment_complete();
				/* translators: %s: is Transaction reference */
				$order->add_order_note( sprintf( __( 'Payment was complete via Adyen (PSP Reference: %s)', 'wkwc-adyen' ), $this->adyen_helper->wkwc_adyen_get_adyen_transaction_url( $reference ) ) );
			} else {
				$order->update_status( 'on-hold', __( 'Payment was successful via Adyen', 'wkwc-adyen' ) );
			}
		}

		return $result;
	}

	/**
	 * Adds an array of fields to be displayed on the gateway's settings screen.
	 *
	 * @since 1.0.3
	 * @return void
	 */
	public function init_form_fields() {

		parent::init_form_fields();

		// only for world wide cards
		if ( 'scheme' === $this->payment_method_type() ) {
			$this->form_fields = array_merge(
				$this->form_fields,
				array(
					'payment_mode'             => array(
						'title'       => esc_html__( 'Payment Mode', 'wkwc-adyen' ),
						'type'        => 'select',
						'label'       => esc_html__( 'Select the Payment Mode', 'wkwc-adyen' ),
						'description' => esc_html__( 'Select Either you want to you authorize payment or by Payment Link ( Redirects to Adyen Payment site.)', 'wkwc-adyen' ),
						'options'     => array(
							'authorize_payment' => esc_html__( 'Authorize Payment', 'wkwc-adyen' ),
							'payment_links'     => esc_html__( 'Payment Links', 'wkwc-adyen' ),
						),
						'default'     => 'authorize_payment',
					),
					'payment_link_desc'        => array(
						'title'   => __( 'Payment Link Description', 'wkwc-adyen' ),
						'type'    => 'text',
						'default' => __( 'Pay with your credit card via Adyen.', 'wkwc-adyen' ),
					),
					'allow_installments'       => array(
						'title'    => __( 'Allow Installments', 'wkwc-adyen' ),
						'label'    => __( 'Yes', 'wkwc-adyen' ),
						'type'     => 'checkbox',
						'default'  => 'no',
						'desc_tip' => __( 'Whether or not to allow installments. This is only for Brazil, Mexico and Turkey', 'wkwc-adyen' ),
					),
					'installments'             => array(
						'title'    => __( 'Installments number', 'wkwc-adyen' ),
						'type'     => 'number',
						'default'  => '20',
						'desc_tip' => __( 'The maximum number for installments (default: 20).', 'wkwc-adyen' ),
					),
					'require_card_holder_name' => array(
						'title'    => __( 'Require card holder name', 'wkwc-adyen' ),
						'label'    => __( 'Yes', 'wkwc-adyen' ),
						'type'     => 'checkbox',
						'default'  => 'yes',
						'desc_tip' => __( 'When checked the card holder field is not required.', 'wkwc-adyen' ),
					),
				)
			);
		}
	}

	/**
	 * Checks if checkout contains at least one subscription.
	 *
	 * @since 1.0.9 - added support for variable subscription
	 *              - change name to `has_subscription`
	 * @since 1.0.3
	 * @return bool
	 */
	public static function has_subscription() {

		if ( WC()->cart ) {

			foreach ( WC()->cart->get_cart() as $item ) {
				if ( $item['data']->is_type( array( 'subscription_variation', 'subscription' ) ) ) {
					return true;
				}
			}
		}

		return false;
	}
}
