<?php
/**
 * Klarna (Pay Now)
 *
 * Payment type     : Buy Now Pay Now
 * Payment flow     : Redirect
 * Countries        : AT, DE, SE, CH, NL
 * Currencies       : EUR, SEK, CHF
 */

namespace WkwcAdyen\Includes\Payment_Methods;

// prevent direct access data leaks
defined( 'ABSPATH' ) || exit;

use Automattic\WooCommerce\StoreApi\Utilities\NoticeHandler;

use WkwcAdyen\Helper\Common\Wkwc_Adyen_Service_Util;

use WkwcAdyen\Helper\Common\Wkwc_Adyen_Checkout_Action;

class Wkwc_Adyen_Klarna_PayNow extends Wkwc_Adyen_Abstract_Gateway {

	/**
	 * $checkout_action
	 *
	 * @var Wkwc_Adyen_Checkout_Action
	 */
	public $checkout_action;

	/**
	 * Constructor of this class.
	 *
	 * @param bool $init_hooks
	 * @since 1.1.0
	 */
	public function __construct( $init_hooks = true ) {

		parent::__construct( $init_hooks );

		$this->has_fields = false;

		if ( 'yes' !== get_option( WKWC_ADN_PREFIX . '_auto_klarna_payments' ) ) {

			$this->supports = array(
				'products',
				'refunds',
			);

		}

		$this->checkout_action = new Wkwc_Adyen_Checkout_Action();
	}

	/**
	 * List of countries where is available.
	 *
	 * show supported countries for recurring payments.
	 *
	 * @return array
	 */
	public function available_countries() {

		return array(
			'AT' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => false,
			),
			'BE' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => true,
			),
			'DE' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => true,
			),
			'SE' => array(
				'currencies' => array( 'SEK' ),
				'recurring'  => true,
			),
			'CH' => array(
				'currencies' => array( 'CHF' ),
				'recurring'  => false,
			),
			'NL' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => true,
			),
		);
	}


	/**
	 * Gets default payment method title.
	 *
	 * @since 1.1.0
	 * @return string
	 */
	public function get_default_title() {
		return __( 'Adyen - Klarna - Pay now', 'wkwc-adyen' );
	}


	/**
	 * Gets default payment method description.
	 *
	 * @since 1.1.0
	 * @return string
	 */
	public function get_default_description() {

		$output  = sprintf( __( 'Complete the full payment instantly through online banking or direct debit. %s', 'wkwc-adyen' ), '<br/>' . $this->show_supported_country() );
		$output .= '<br/>' . $this->show_rec_supported_country();

		return $output;
	}

	/**
	 * Gets default description set in settings.
	 *
	 * @since 1.1.0
	 * @return string
	 */
	public function get_settings_description() {
		$this->description = ( $this->settings['description'] ) ? $this->settings['description'] : __( 'Pay with - Klarna Pay Now', 'wkwc-adyen' );
		return $this->description;
	}


	/**
	 * Type of the payment method (e.g ideal, scheme. bcmc).
	 *
	 * @since 1.1.0
	 * @return string
	 */
	public function payment_method_type() {
		return 'klarna_paynow';
	}

	/**
	 * Returns the payment method to be used for recurring payments
	 *
	 * @since 1.1.0
	 * @return string
	 */
	public function recurring_payment_method() {
		return $this->payment_method_type();
	}

	/**
	 * Validates extra added fields.
	 *
	 * @since 1.1.0
	 * @return bool
	 */
	public function validate_fields() {
		return parent::validate_fields();
	}

	/**
	 * Add payment fields.
	 *
	 * @since 1.0.0
	 * @return void
	 */
	public function payment_fields() {

		parent::payment_fields();

		echo $this->generate_extra_fields_html();
	}

	/**
	 * Generates extra fields HTML.
	 *
	 * @since 1.1.0 - added CSS class
	 * @since 1.0.0
	 * @return string
	 */
	public function generate_extra_fields_html() {
		?>
		<div id="<?php echo esc_attr( WKWC_ADN_PREFIX . '-' . $this->payment_method_type() . '-container' ); ?>"></div>
		<input type="hidden" id="<?php echo esc_attr( 'wkwc-adyen' . $this->payment_method_type() . '_issuer' ); ?>" name="<?php echo esc_attr( WKWC_ADN_PREFIX . '_' . $this->payment_method_type() . '_issuer' ); ?>">
		<?php
	}

	/**
	 * Builds the required payment payload
	 *
	 * @since 1.1.0
	 * @param \WC_Order $order
	 * @param string    $reference
	 * @return array
	 */
	public function build_payment_payload( \WC_Order $order, $reference ) {
		return parent::build_payment_payload( $order, $reference );
	}

	/**
	 * Processes the payment.
	 *
	 * \WC_Order instance to manipulate metadata
	 *
	 * @param int $order_id
	 * @return array
	 */
	public function process_payment( $order_id ) {

		parent::process_payment( $order_id );

		$order = wc_get_order( $order_id );

		$reference = $order_id;

		$payload = $this->build_payment_payload( $order, $reference );

		$response = $this->checkout_action->send_payment( $payload );

		if ( $response->status == 200 ) {

			return $this->process_payment_result( $response, $order );

		} else {

			wc_add_notice( $response->body['message'], 'error' );

			NoticeHandler::convert_notices_to_exceptions( 'woocommerce_rest_payment_error' );

		}

		return array( 'result' => 'failure' );
	}

	/**
	 * Processes the payment result.
	 *
	 * @param object    $response
	 * @param \WC_Order $order
	 * @return array
	 */
	protected function process_payment_result( $response, $order ) {
		$body_response = $this->adyen_helper::wkwc_adyen_obj_to_arr( $response->body );
		$result_code   = $body_response['resultCode'];
		$action        = $body_response['action'];

		$extra_action_codes = array( 'RedirectShopper' );
		$error_codes        = array( 'Refused', 'Error', 'Cancelled' );

		$result = array(
			'result'   => 'success',
			'redirect' => Wkwc_Adyen_Service_Util::get_return_page_url( $order, $result_code ),
		);

		$order->update_meta_data( '_' . WKWC_ADN_PREFIX . '_payment_resultCode', $result_code );
		$order->update_meta_data( '_' . WKWC_ADN_PREFIX . '_payment_action', $action );
		$order->save();

		if ( in_array( $result_code, $extra_action_codes ) ) {

			// redirect to process payment action via Web Component
			$result = array(
				'result'   => 'success',
				'redirect' => add_query_arg(
					array(
						WKWC_ADN_PREFIX . '_payment_method' => $this->payment_method_type(),
						WKWC_ADN_PREFIX . '_order_id' => $order->get_id(),
					),
					Wkwc_Adyen_Service_Util::get_checkout_url( $order )
				),
			);

		} elseif ( in_array( $result_code, $error_codes ) ) {

			wc_add_notice( __( 'The transaction could not be processed! Either it has been refused or an error has occurred! Please make sure you provide a valid bank account.', 'integration-adyen-woocommerce' ), 'error' );

			NoticeHandler::convert_notices_to_exceptions( 'woocommerce_rest_payment_error' );

			$result = array(
				'result' => 'failure',
			);

		}

		return $result;
	}

	/**
	 * Checks whether or not the recurring payments are supported by the country.
	 *
	 * @since 1.2.0
	 * @return bool
	 */
	public function support_recurring() {

		if ( WC()->cart && $this->get_order_total() > 0 ) {

			if ( ! empty( $this->available_countries() ) ) {
				$available_countries = $this->available_countries();
				$customer_country    = WC()->customer->get_billing_country();
				$country             = $available_countries[ $customer_country ];

				return $country['recurring'] === true ? true : false;
			}
		}

		return true;
	}

	/**
	 * Display the countries which support recurring payments.
	 *
	 * @since 1.2.0
	 * @return array
	 */
	public function show_rec_supported_country() {

		$output    = '';
		$countries = array();

		foreach ( $this->available_countries() as $country_code => $data ) {

			$country_code = '_ANY_' === $country_code ? __( 'ANY', 'wkwc-adyen' ) : $country_code;

			if ( $data['recurring'] === true ) {
				if ( empty( $data['currencies'] ) ) {
					$countries[] = $country_code;
				} else {
					$currencies  = $data['currencies'];
					$countries[] = $country_code . ' (' . implode( ', ', $currencies ) . ')';
				}
			}
		}

		if ( ! empty( $countries ) ) {
			$output = sprintf( __( '%1$sSupported country for recurring payments:%2$s %3$s', 'wkwc-adyen' ), '<b>', '</b>', implode( ', ', $countries ) );
		}

		return $output;
	}
}
