<?php
/**
 * Trustly
 *
 * Payment type     : Online Banking
 * Payment flow     : Redirect
 * Countries        : NO, LV, GB, FI, CZ, DK, SK, SE, ES, LT, EE
 * Currencies       : DKK, EUR, NOK, PLN, SEK
 * Recurring        : Only in Sweden
 */

namespace WkwcAdyen\Includes\Payment_Methods;

// prevent direct access data leaks
defined( 'ABSPATH' ) || exit;

use Automattic\WooCommerce\StoreApi\Utilities\NoticeHandler;

use WkwcAdyen\Helper\Common\Wkwc_Adyen_Service_Util;

use WkwcAdyen\Helper\Common\Wkwc_Adyen_Checkout_Action;

class Wkwc_Adyen_Trustly extends Wkwc_Adyen_Abstract_Gateway {

	/**
	 * $checkout_action
	 *
	 * @var Wkwc_Adyen_Checkout_Action
	 */
	public $checkout_action;

	/**
	 * Constructor of this class.
	 *
	 * @param bool $init_hooks
	 * @since 1.1.0
	 */
	public function __construct( $init_hooks = true ) {

		parent::__construct( $init_hooks );

		$this->has_fields = false;

		$this->supports = array(
			'products',
			'refunds',
		);

		$this->checkout_action = new Wkwc_Adyen_Checkout_Action();
	}

	/**
	 * List of countries where is available.
	 *
	 * @return array
	 */
	public function available_countries() {

		return array(
			'NO' => array(
				'currencies' => array( 'NOK' ),
				'recurring'  => false,
			),
			'LV' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => false,
			),
			'GB' => array(
				'currencies' => array( 'GBP' ),
				'recurring'  => false,
			),
			'FI' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => false,
			),
			'CZ' => array(
				'currencies' => array( 'CZK' ),
				'recurring'  => false,
			),
			'DK' => array(
				'currencies' => array( 'DKK' ),
				'recurring'  => false,
			),
			'SK' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => false,
			),
			'SE' => array(
				'currencies' => array( 'SEK' ),
				'recurring'  => true,
			),
			'ES' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => false,
			),
			'LT' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => false,
			),
			'EE' => array(
				'currencies' => array( 'EUR' ),
				'recurring'  => false,
			),
		);
	}

	/**
	 * Gets default payment method title.
	 *
	 * @return string
	 */
	public function get_default_title() {
		return __( 'Adyen - Trustly', 'integration-adyen-woocommerce' );
	}



	/**
	 * Gets default payment method description.
	 *
	 * @return string
	 */
	public function get_default_description() {

		$output = sprintf( __( 'Shoppers can pay with Trustly when shopping online or in-store using our terminals. %s', 'integration-adyen-woocommerce' ), '<br/>' . $this->show_supported_country() );

		return $output;
	}

	/**
	 * Type of the payment method (e.g ideal, scheme. bcmc).
	 *
	 * @return string
	 */
	public function payment_method_type() {
		return 'trustly';
	}

	/**
	 * Gets default description set in settings.
	 *
	 * @since 1.1.0
	 * @return string
	 */
	public function get_settings_description() {
		$this->description = ( $this->settings['description'] ) ? $this->settings['description'] : __( 'Pay with - Trustly', 'wkwc-adyen' );
		return $this->description;
	}

	/**
	 * Returns the payment method to be used for recurring payments
	 *
	 * @since 1.1.0
	 * @return string
	 */
	public function recurring_payment_method() {
		return $this->payment_method_type();
	}

	/**
	 * Validates extra added fields.
	 *
	 * @since 1.1.0
	 * @return bool
	 */
	public function validate_fields() {
		return parent::validate_fields();
	}

	/**
	 * Adds an array of fields to be displayed on the gateway's settings screen.
	 *
	 * @since 1.1.0
	 * @return void
	 */
	public function init_form_fields() {

		parent::init_form_fields();
	}


	/**
	 * Builds the required payment payload
	 *
	 * @since 1.1.0
	 * @param \WC_Order $order
	 * @param string    $reference
	 * @return array
	 */
	public function build_payment_payload( \WC_Order $order, $reference ) {
		return parent::build_payment_payload( $order, $reference );
	}

	/**
	 * Processes the payment.
	 *
	 * @since 1.1.0
	 * @param int $order_id
	 * @return array
	 */
	public function process_payment( $order_id ) {

		parent::process_payment( $order_id );

		$order = wc_get_order( $order_id );

		$payload = $this->build_payment_payload( $order, $order_id );

		$response = $this->checkout_action->send_payment( $payload );

		if ( $response->status == 200 ) {

			return $this->process_payment_result( $response, $order );

		} else {

			wc_add_notice( $response->body['message'], 'error' );

			NoticeHandler::convert_notices_to_exceptions( 'woocommerce_rest_payment_error' );

		}

		return array( 'result' => 'failure' );
	}


	/**
	 * Processes the payment result.
	 *
	 * @since 1.2.0
	 * @param object    $response
	 * @param \WC_Order $order
	 * @return array
	 */
	protected function process_payment_result( $response, $order ) {
		$body_response = $this->adyen_helper::wkwc_adyen_obj_to_arr( $response->body );
		$result_code   = $body_response['resultCode'];
		$action        = $body_response['action'];

		$result = array(
			'result'   => 'success',
			'redirect' => Wkwc_Adyen_Service_Util::get_return_page_url( $order, $result_code ),
		);

		$order->update_meta_data( '_' . WKWC_ADN_PREFIX . '_payment_resultCode', $result_code );
		$order->update_meta_data( '_' . WKWC_ADN_PREFIX . '_payment_action', $action );
		$order->save();

		if ( 'RedirectShopper' == $result_code ) {

			// redirect to process payment action via Web Component
			$result = array(
				'result'   => 'success',
				'redirect' => add_query_arg(
					array(
						WKWC_ADN_PREFIX . '_payment_method' => $this->payment_method_type(),
						WKWC_ADN_PREFIX . '_order_id' => $order->get_id(),
					),
					Wkwc_Adyen_Service_Util::get_checkout_url( $order )
				),
			);

		}

		return $result;
	}
}
