<?php
/**
 * This file handles user related functions

 * @package WkwcAdyen
 * @version 1.0.0
 */

namespace WkwcAdyen\Rest_Api\Includes\Front\Checkout;

defined( 'ABSPATH' ) || exit;

use WC_Order;
use Adyen\Service\ResourceModel\Payment\Authorise;
use Adyen\Service\Checkout;
use WkwcAdyen\Helper\Adyen\Wkwc_Adyen_Helper;

if ( ! class_exists( 'Wkwc_Adyen_Execute_Checkout' ) ) {
	/**
	 * Users class
	 */
	class Wkwc_Adyen_Execute_Checkout {

		/**
		 * $adyen_config
		 *
		 * @var object
		 */
		protected $adyen_config;

		/**
		 * $adyen_ctrl
		 *
		 * @var object
		 */
		protected $adyen_ctrl;

		/**
		 * Construct
		 *
		 * @return void
		 */
		public function __construct() {
			$this->adyen_ctrl   = new Wkwc_Adyen_Helper();
			$this->adyen_config = $this->adyen_ctrl->adyen_config;
		}

		/**
		 * Get Merchant Payment Method.
		 *
		 * @param array $request API params.
		 *
		 * @return mixed
		 */
		public function wkwc_adyen_get_payment_method( $request ) {
			$request = is_object( $request ) ? $request->get_params() : $request;
			$url     = '/paymentMethods';

			$params = array(
				'countryCode'     => 'IN',
				'shopperLocale'   => 'IN',
				'amount'          => array(
					'currency' => $request['currency'],
					'value'    => $request['amount'],
				),
				'channel'         => 'Web',
				'merchantAccount' => $this->adyen_config->wkwc_adyen_get_merchantcode(),
			);

			$result = $this->adyen_ctrl->wkwc_adyen_remote_request( $url, $params, 'POST' );

			return rest_ensure_response( $result );
		}

		/**
		 * Create New Payment Method
		 *
		 * @param array $request api params.
		 * @return mixed
		 */
		public function wkwc_adyen_create_new_payment( $request ) {

			$request  = is_object( $request ) ? $request->get_params() : $request;
			$order_id = $request['order_id'];

			try {
				$service = new Checkout( $this->adyen_ctrl->wkwc_adyen_get_client() );
			} catch ( AdyenException $exception ) {
				wkwc_adn_log( $exception->getMessage() );
				return array();
			}

			$url = '/payments';

			$params = array(
				'paymentMethod'   => $request['paymentMethod'],
				'amount'          => array(
					'currency' => $request['amount']['currency'],
					'value'    => $request['amount']['value'],
				),
				'reference'       => $order_id,
				'returnUrl'       => $request['returnUrl'],
				'merchantAccount' => $this->adyen_config->wkwc_adyen_get_merchantcode(),
			);

			$result = $this->adyen_ctrl->wkwc_adyen_remote_request( $url, $params, 'POST' );

			return rest_ensure_response( $result );
		}

		public function wkwc_adyen_get_return_url_response( $request ) {
			$request = is_object( $request ) ? $request->get_params() : $request;

			wkwc_adn_log( 'Return result response data: ' . print_r( $request, true ) );
		}

		/**
		 * Get Created Payment Info Method
		 *
		 * @param array $request api params.
		 * @return mixed
		 */
		public function wkwc_adyen_get_payment_info( $request ) {
			$request = is_object( $request ) ? $request->get_params() : $request;
			$url     = '/payments/details';

			$params = array(
				'details'     => array(
					'PaRes' => $request['payment_reference'],
				),
				'paymentData' => '',
			);

			$result = $this->adyen_ctrl->wkwc_adyen_remote_request( $url, $params, 'POST' );

			return rest_ensure_response( $result );
		}

		/**
		 * Create Payment Link Method.
		 *
		 * @param array $request API params.
		 *
		 * @return mixed
		 */
		public function wkwc_adyen_create_payment_link( $request ) {
			$request  = is_object( $request ) ? $request->get_params() : $request;
			$url      = '/paymentLinks';
			$order_id = $request['order_id'];
			$order    = new WC_Order( $order_id );
			$shipping = $order->get_shipping_methods();
			$name     = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();

			$billing_address = array(
				'street'            => $order->get_billing_address_1(),
				'postalCode'        => $order->get_billing_postcode(),
				'city'              => $order->get_billing_city(),
				'houseNumberOrName' => $name,
				'stateOrProvince'   => $order->get_billing_state(),
				'country'           => $order->get_billing_country(),
			);

			if ( empty( $shipping ) ) {
				$shipping_address = $billing_address;
			} else {
				$shipping_address = array(
					'postalCode'        => $order->get_shipping_postcode(),
					'city'              => $order->get_shipping_city(),
					'houseNumberOrName' => $name,
					'street'            => $order->get_shipping_address_1(),
					'stateOrProvince'   => $order->get_shipping_state(),
					'country'           => $order->get_shipping_country(),
				);
			}

			$params = array(
				'reference'        => strval( $order_id ),
				'amount'           => array(
					'value'    => $order->get_total() * 100,
					'currency' => get_woocommerce_currency(),
				),
				'merchantAccount'  => $this->adyen_config->wkwc_adyen_get_merchantcode(),
				'countryCode'      => $order->get_shipping_country(),
				'shopperReference' => 'wkcust_ref-' . $order->get_customer_id(),
				'shopperEmail'     => $order->get_billing_email(),
				'shopperLocale'    => $order->get_shipping_country(),
				'billingAddress'   => $billing_address,
				'deliveryAddress'  => $shipping_address,
			);

			$result = $this->adyen_ctrl->wkwc_adyen_remote_request( $url, $params, 'POST' );

			return rest_ensure_response( $result );
		}

		/**
		 * Get Created Payment Link Method
		 *
		 * @param array $request api params.
		 * @return mixed
		 */
		public function wkwc_adyen_get_payment_link_info( $request ) {
			$request = is_object( $request ) ? $request->get_params( 'payment_link_id' ) : $request;
			$link_id = $request['payment_link_id'];
			$url     = '/paymentLinks/' . $link_id;
			$result  = $this->adyen_ctrl->wkwc_adyen_remote_request( $url, '', 'GET' );

			return rest_ensure_response( $result );
		}

		/**
		 * Update Payment Status Method
		 *
		 * @param array $request api params.
		 * @return mixed
		 */
		public function wkwc_adyen_update_payment_link_info( $request ) {
			$request = is_object( $request ) ? $request->get_params( 'payment_link_id' ) : $request;
			$link_id = $request['payment_link_id'];

			$params = array(
				'status' => $request['link_status'],
			);

			$url = '/paymentLinks/' . $link_id;

			$result = $this->adyen_ctrl->wkwc_adyen_remote_request( $url, $params, 'PATCH' );

			return rest_ensure_response( $result );
		}
	}
}
