<?php
/**
 * Plugin Name: WAO Debounce Whitelist Multisite
 * Description: Permite definir, a nivel de red (multisite), una lista de dominios y correos que no deben ser bloqueados por la validación (ej. DeBounce) en el checkout de WooCommerce.
 * Author: Maykel J. Mdez
 * Version: 1.0
 * Network: true
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Nombre de las opciones de red
 */
const DWM_OPTION_DOMAINS = 'dwm_whitelist_domains';
const DWM_OPTION_EMAILS  = 'dwm_whitelist_emails';

/**
 * Añadir página en el Network Admin
 */
add_action( 'network_admin_menu', function() {

    // Solo super admins
    if ( ! current_user_can( 'manage_network_options' ) ) {
        return;
    }

    add_submenu_page(
        'settings.php',                          // Menú padre (Ajustes de red)
        'Debounce Whitelist',                    // Título de la página
        'Debounce Whitelist',                    // Texto del menú
        'manage_network_options',                // Capacidad
        'dwm-debounce-whitelist',                // slug
        'dwm_render_network_settings_page'       // Callback
    );
} );

/**
 * Renderizar la página de ajustes en la red
 */
function dwm_render_network_settings_page() {

    if ( ! current_user_can( 'manage_network_options' ) ) {
        wp_die( 'No tienes permisos suficientes para acceder a esta página.' );
    }

    // Procesar guardado
    if ( isset( $_POST['dwm_save_settings'] ) && check_admin_referer( 'dwm_save_settings_action', 'dwm_nonce_field' ) ) {

        // Dominios
        $domains_raw = isset( $_POST['dwm_domains'] ) ? wp_unslash( $_POST['dwm_domains'] ) : '';
        $domains_arr = array();

        if ( ! empty( $domains_raw ) ) {
            $lines = preg_split( '/\r\n|\r|\n/', $domains_raw );
            foreach ( $lines as $line ) {
                $line = trim( strtolower( $line ) );
                if ( $line !== '' ) {
                    // Por si el usuario pone el dominio con @ delante, lo quitamos
                    if ( strpos( $line, '@' ) !== false ) {
                        $line = substr( strrchr( $line, '@' ), 1 );
                        $line = trim( $line );
                    }
                    if ( $line !== '' ) {
                        $domains_arr[] = $line;
                    }
                }
            }
            $domains_arr = array_unique( $domains_arr );
        }

        // Emails
        $emails_raw = isset( $_POST['dwm_emails'] ) ? wp_unslash( $_POST['dwm_emails'] ) : '';
        $emails_arr = array();

        if ( ! empty( $emails_raw ) ) {
            $lines = preg_split( '/\r\n|\r|\n/', $emails_raw );
            foreach ( $lines as $line ) {
                $line = trim( strtolower( $line ) );
                if ( $line !== '' ) {
                    $emails_arr[] = $line;
                }
            }
            $emails_arr = array_unique( $emails_arr );
        }

        // Guardar en opciones de red
        update_site_option( DWM_OPTION_DOMAINS, $domains_arr );
        update_site_option( DWM_OPTION_EMAILS,  $emails_arr );

        ?>
        <div class="notice notice-success is-dismissible">
            <p>Configuración guardada correctamente.</p>
        </div>
        <?php
    }

    // Obtener valores actuales
    $saved_domains = (array) get_site_option( DWM_OPTION_DOMAINS, array() );
    $saved_emails  = (array) get_site_option( DWM_OPTION_EMAILS, array() );

    $domains_textarea = implode( "\n", $saved_domains );
    $emails_textarea  = implode( "\n", $saved_emails );

    ?>
    <div class="wrap">
        <h1>Debounce Whitelist (Multisite)</h1>

        <p>
            Aquí puedes definir una lista de <strong>dominios</strong> y <strong>correos electrónicos</strong> que
            no deben ser bloqueados por la validación (por ejemplo, DeBounce) en el checkout de WooCommerce, en
            <strong>todos los sitios</strong> de la red.
        </p>

        <form method="post">
            <?php wp_nonce_field( 'dwm_save_settings_action', 'dwm_nonce_field' ); ?>
            <input type="hidden" name="dwm_save_settings" value="1" />

            <table class="form-table" role="presentation">
                <tr>
                    <th scope="row">
                        <label for="dwm_domains">Dominios en whitelist</label>
                    </th>
                    <td>
                        <textarea
                            name="dwm_domains"
                            id="dwm_domains"
                            rows="10"
                            cols="50"
                            class="large-text code"
                        ><?php echo esc_textarea( $domains_textarea ); ?></textarea>
                        <p class="description">
                            Uno por línea, <strong>sin @</strong>. Ejemplo:<br>
                            <code>midominio.com</code><br>
                            <code>clientevip.com</code>
                        </p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="dwm_emails">Correos en whitelist</label>
                    </th>
                    <td>
                        <textarea
                            name="dwm_emails"
                            id="dwm_emails"
                            rows="10"
                            cols="50"
                            class="large-text code"
                        ><?php echo esc_textarea( $emails_textarea ); ?></textarea>
                        <p class="description">
                            Uno por línea, email completo. Ejemplo:<br>
                            <code>prueba@midominio.com</code><br>
                            <code>soporte@clientevip.com</code>
                        </p>
                    </td>
                </tr>
            </table>

            <?php submit_button( 'Guardar cambios' ); ?>
        </form>
    </div>
    <?php
}

/**
 * Lógica de saltar errores de email (incluyendo DeBounce)
 * SOLO en checkout de WooCommerce, usando la whitelist de red.
 */
add_action( 'woocommerce_after_checkout_validation', function( $data, $errors ) {

    if ( empty( $data['billing_email'] ) ) {
        return;
    }

    // Leer la whitelist desde opciones de RED
    $whitelist_domains = (array) get_site_option( DWM_OPTION_DOMAINS, array() );
    $whitelist_emails  = (array) get_site_option( DWM_OPTION_EMAILS, array() );

    // Normalizar arrays (por si acaso)
    $whitelist_domains = array_map( 'strtolower', array_map( 'trim', $whitelist_domains ) );
    $whitelist_emails  = array_map( 'strtolower', array_map( 'trim', $whitelist_emails ) );

    $email  = strtolower( trim( $data['billing_email'] ) );
    $domain = substr( strrchr( $email, '@' ), 1 );

    $skip = false;

    if ( in_array( $email, $whitelist_emails, true ) ) {
        $skip = true;
    }

    if ( in_array( $domain, $whitelist_domains, true ) ) {
        $skip = true;
    }

    // Si NO está en whitelist → no tocamos nada, que DeBounce valide normal.
    if ( ! $skip ) {
        return;
    }

    // === 1) Limpiar errores de email en el objeto WP_Error ===
    if ( is_wp_error( $errors ) ) {

        foreach ( $errors->get_error_codes() as $code ) {

            $messages    = $errors->get_error_messages( $code );
            $remove_code = false;

            foreach ( $messages as $msg ) {
                if (
                    stripos( $msg, 'email' )  !== false ||
                    stripos( $msg, 'correo' ) !== false ||
                    stripos( $msg, 'e-mail' ) !== false ||
                    stripos( $msg, 'mail' )   !== false
                ) {
                    $remove_code = true;
                    break;
                }
            }

            if ( $remove_code ) {
                $errors->remove( $code );
            }
        }
    }

    // === 2) Limpiar errores de email en las notices de WooCommerce ===
    if ( function_exists( 'wc_get_notices' ) && function_exists( 'wc_clear_notices' ) && function_exists( 'wc_add_notice' ) ) {

        $notices = wc_get_notices();

        if ( ! empty( $notices['error'] ) && is_array( $notices['error'] ) ) {

            $new_errors = array();

            foreach ( $notices['error'] as $notice ) {

                $msg = is_array( $notice ) && isset( $notice['notice'] ) ? $notice['notice'] : $notice;

                if (
                    stripos( $msg, 'email' )  !== false ||
                    stripos( $msg, 'correo' ) !== false ||
                    stripos( $msg, 'e-mail' ) !== false ||
                    stripos( $msg, 'mail' )   !== false
                ) {
                    // Parece error de email → lo saltamos
                    continue;
                }

                // Otros errores (nombre, dirección, etc.) se mantienen
                $new_errors[] = $notice;
            }

            wc_clear_notices();

            foreach ( $new_errors as $notice ) {
                $msg = is_array( $notice ) && isset( $notice['notice'] ) ? $notice['notice'] : $notice;
                wc_add_notice( $msg, 'error' );
            }
        }
    }

}, PHP_INT_MAX, 2 );
