<?php
/*
Plugin Name: User IP Restriction
Description: Restrict users to log in only from specific IPs.
Version: 1.2
Author: Maykel Jose Menendez
*/

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

class UserIPRestriction {
    public function __construct() {
        if (is_multisite() && is_network_admin()) {
            add_action('network_admin_menu', [$this, 'add_network_admin_menu']);
        } else {
            add_action('admin_menu', [$this, 'add_admin_menu']);
        }
        add_action('admin_init', [$this, 'check_admin_access']);
        add_action('admin_init', [$this, 'save_ip_restrictions']);
        add_action('wp_login', [$this, 'validate_user_ip_on_login'], 10, 2);
    }

    public function add_network_admin_menu() {
        add_menu_page(
            'User IP Restriction',
            'IP Restriction',
            'manage_network_options',
            'user-ip-restriction',
            [$this, 'settings_page'],
            'dashicons-shield-alt',
            80
        );
    }

    public function add_admin_menu() {
        if (!is_multisite()) {
            add_menu_page(
                'User IP Restriction',
                'IP Restriction',
                'manage_options',
                'user-ip-restriction',
                [$this, 'settings_page'],
                'dashicons-shield-alt',
                80
            );
        }
    }

    public function settings_page() {
        $users = get_users();
        $restrictions = get_site_option('user_ip_restrictions', []);
        ?>
        <div class="wrap">
            <h1>User IP Restriction</h1>
            <style>
                form.form-table {
                    background-color: #f9f9f9;
                    border: 1px solid #e1e1e1;
                    padding: 20px;
                    border-radius: 5px;
                    margin-top: 20px;
                }
                form.form-table label {
                    font-weight: bold;
                }
                form.form-table input[type="text"],
                form.form-table select {
                    padding: 8px;
                    font-size: 14px;
                    border: 1px solid #ddd;
                    border-radius: 3px;
                    width: calc(100% - 16px);
                    margin-bottom: 10px;
                }
                form.form-table button.button-primary {
                    background-color: #0073aa;
                    color: white;
                    border: none;
                    padding: 10px 20px;
                    border-radius: 3px;
                    cursor: pointer;
                }
                form.form-table button.button-primary:hover {
                    background-color: #005f8d;
                }
            </style>
            <form method="post" class="form-table">
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="user">Select User</label></th>
                        <td>
                            <select name="user" id="user" required>
                                <option value="">Select a User</option>
                                <?php foreach ($users as $user): ?>
                                    <option value="<?php echo esc_attr($user->ID); ?>"><?php echo esc_html($user->display_name); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="ip">IP Addresses or CIDR Blocks (separado por coma. IP1,IP2)</label></th>
                        <td><input type="text" name="ip" id="ip" required placeholder="Enter IPs separated by commas (e.g., 1.1.1.1, 1.1.1.0/24)"></td>
                    </tr>
                </table>
                <p class="submit">
                    <button type="submit" name="add_ip_restriction" class="button button-primary">Add Restriction</button>
                </p>
            </form>

            <h2>Current Restrictions</h2>
            <style>
                table.widefat {
                    border-collapse: collapse;
                    width: 100%;
                    margin-top: 20px;
                    background-color: #f9f9f9;
                    border: 1px solid #e1e1e1;
                }
                table.widefat th {
                    background-color: #f1f1f1;
                    padding: 10px;
                    text-align: left;
                    border-bottom: 2px solid #e1e1e1;
                }
                table.widefat td {
                    padding: 10px;
                    border-bottom: 1px solid #e1e1e1;
                }
                table.widefat tr:hover {
                    background-color: #f3f3f3;
                }
                button.button-secondary {
                    background-color: #0073aa;
                    color: #fff;
                    border: none;
                    padding: 5px 10px;
                    border-radius: 3px;
                    cursor: pointer;
                }
                button.button-secondary:hover {
                    background-color: #005f8d;
                }
            </style>
            <table class="widefat">
                <thead>
                <tr>
                    <th>User</th>
                    <th>IP Addresses or CIDR Blocks</th>
                    <th>Actions</th>
                </tr>
                </thead>
                <tbody>
                <?php if (!empty($restrictions)): ?>
                    <?php foreach ($restrictions as $index => $restriction): ?>
                        <tr>
                            <td><?php echo esc_html(get_userdata($restriction['user'])->display_name); ?></td>
                            <td>
                                <form method="post" style="display: flex; align-items: center;">
                                    <input type="text" name="edit_ip" value="<?php echo esc_attr($restriction['ip']); ?>" style="flex: 1; margin-right: 10px;">
                                    <input type="hidden" name="edit_index" value="<?php echo esc_attr($index); ?>">
                                    <button type="submit" name="update_ip_restriction" class="button button-secondary">Update</button>
                                </form>
                            </td>
                            <td>
                                <form method="post" style="display:inline;">
                                    <input type="hidden" name="delete_index" value="<?php echo esc_attr($index); ?>">
                                    <button type="submit" name="delete_ip_restriction" class="button button-secondary">Delete</button>
                                </form>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php else: ?>
                    <tr>
                        <td colspan="3">No restrictions found.</td>
                    </tr>
                <?php endif; ?>
                </tbody>
            </table>
        </div>
        <?php
    }

    public function save_ip_restrictions() {
        if (isset($_POST['add_ip_restriction']) && current_user_can('manage_network_options')) {
            $user_id = intval($_POST['user']);
            $ip = sanitize_text_field($_POST['ip']);

            if ($user_id && $ip) {
                $restrictions = get_site_option('user_ip_restrictions', []);
                $restrictions[] = [
                    'user' => $user_id,
                    'ip' => $ip
                ];
                update_site_option('user_ip_restrictions', $restrictions);
            }
        }

        if (isset($_POST['delete_ip_restriction']) && current_user_can('manage_network_options')) {
            $index = intval($_POST['delete_index']);
            $restrictions = get_site_option('user_ip_restrictions', []);

            if (isset($restrictions[$index])) {
                unset($restrictions[$index]);
                update_site_option('user_ip_restrictions', array_values($restrictions));
            }
        }

        if (isset($_POST['update_ip_restriction']) && current_user_can('manage_network_options')) {
            $index = intval($_POST['edit_index']);
            $new_ip = sanitize_text_field($_POST['edit_ip']);
            $restrictions = get_site_option('user_ip_restrictions', []);

            if (isset($restrictions[$index]) && $new_ip) {
                $restrictions[$index]['ip'] = $new_ip;
                update_site_option('user_ip_restrictions', $restrictions);
            }
        }
    }

    public function validate_user_ip_on_login($user_login, $user) {
        $this->check_user_ip($user->ID);
    }

    public function check_admin_access() {
        if (is_user_logged_in()) {
            $current_user = wp_get_current_user();
            $this->check_user_ip($current_user->ID);
        }
    }

    private function check_user_ip($user_id) {
        $restrictions = get_site_option('user_ip_restrictions', []);
        $user_ip = $_SERVER['REMOTE_ADDR'];

        foreach ($restrictions as $restriction) {
            if ($restriction['user'] == $user_id) {
                $ip_list = explode(',', $restriction['ip']);
                foreach ($ip_list as $ip_range) {
                    if ($this->ip_in_range(trim($user_ip), trim($ip_range))) {
                        return; // La IP es válida, permitir acceso
                    }
                }
                wp_die(__('Access denied: Your IP address is not allowed for this user.', 'user-ip-restriction'));
            }
        }
    }

    private function ip_in_range($ip, $range) {
        if (strpos($range, '/') === false) {
            // Comparar una IP específica
            return $ip === $range;
        } else {
            // Comparar un bloque CIDR
            list($subnet, $bits) = explode('/', $range);
            $ip = ip2long($ip);
            $subnet = ip2long($subnet);
            $mask = -1 << (32 - $bits);
            $subnet &= $mask; // Aplicar máscara
            return ($ip & $mask) === $subnet;
        }
    }
}

new UserIPRestriction();
