<?php
/*
Plugin Name: WAO Multisite Email Blocker
Description: Bloquea correos electrónicos o dominios específicos en el checkout de WooCommerce en toda la red.
Network: true
Version: 1.2
Author: Maykel-J
*/

if (!defined('ABSPATH')) exit;

class Multisite_Email_Blocker {

    const OPTION_KEY = 'blocked_emails_list';

    public function __construct() {
        if (!is_multisite()) return;

        // Página de menú como entrada principal
        add_action('network_admin_menu', [$this, 'add_network_menu']);
        add_action('network_admin_edit_save_blocked_emails', [$this, 'save_blocked_emails']);

        // Validación en WooCommerce checkout
        add_action('woocommerce_checkout_process', [$this, 'validate_checkout_email']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_checkout_script']);
    }

    public function add_network_menu() {
        add_menu_page(
            'Email Blocker',
            'Email Blocker',
            'manage_network_options',
            'email-blocker',
            [$this, 'render_settings_page'],
            'dashicons-email-alt2',
            30
        );
    }

    public function render_settings_page() {
        $emails = get_site_option(self::OPTION_KEY, '');
        ?>
        <div class="wrap">
            <h1>Blocked Emails</h1>
            <form method="post" action="edit.php?action=save_blocked_emails">
                <?php wp_nonce_field('save_blocked_emails_nonce'); ?>
                <textarea name="blocked_emails" rows="10" cols="80" placeholder="email1@example.com, @domain.com"><?php echo esc_textarea($emails); ?></textarea>
                <p class="description">Enter blocked emails or domains separated by comma.<br>
                Examples: <code>spam@example.com</code>, <code>@baddomain.com</code></p>
                <?php submit_button('Save'); ?>
            </form>
        </div>
        <?php
    }

    public function save_blocked_emails() {
        check_admin_referer('save_blocked_emails_nonce');
        if (!current_user_can('manage_network_options')) {
            wp_die(__('You do not have permission to update this setting.'));
        }

        $raw_input = isset($_POST['blocked_emails']) ? sanitize_text_field($_POST['blocked_emails']) : '';
        update_site_option(self::OPTION_KEY, $raw_input);

        wp_redirect(network_admin_url('admin.php?page=email-blocker&updated=1'));
        exit;
    }

    public function get_blocked_items() {
        $raw = get_site_option(self::OPTION_KEY, '');
        $list = array_map('trim', explode(',', $raw));
        return array_filter($list);
    }

    public function validate_checkout_email() {
        if (!function_exists('WC')) return;
        $email = isset($_POST['billing_email']) ? sanitize_email($_POST['billing_email']) : '';
        $blocked = $this->get_blocked_items();

        foreach ($blocked as $entry) {
            if (stripos($email, $entry) !== false) {
                wc_add_notice(__('This email address is blocked.', 'email-blocker'), 'error');
                break;
            }
        }
    }

  public function enqueue_checkout_script() {
    if (!function_exists('is_checkout')) return;
    if (!is_checkout()) return;

    $blocked = $this->get_blocked_items();
    if (empty($blocked)) return;

    $blocked_json = wp_json_encode($blocked);

    $js = <<<JS
jQuery(document).ready(function($) {
    let blocked = $blocked_json;
    let \$email = $("#billing_email");
    let \$form = $("form.checkout");

    function checkBlockedEmail(email) {
        email = email.toLowerCase();
        for (let i = 0; i < blocked.length; i++) {
            if (email.includes(blocked[i].toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    function showError() {
        if (!$(".email-blocked-error").length) {
            \$email.after("<div class='woocommerce-error email-blocked-error'>This email address is blocked.</div>");
        }
    }

    function clearError() {
        $(".email-blocked-error").remove();
    }

    \$email.on("input", function() {
        let val = $(this).val();
        clearError();
        if (checkBlockedEmail(val)) {
            showError();
        }
    });

    \$form.on("checkout_place_order", function(e) {
        let val = \$email.val();
        clearError();
        if (checkBlockedEmail(val)) {
            showError();
            return false;
        }
        return true;
    });
});
JS;

    wp_add_inline_script('jquery', $js);
}

}

new Multisite_Email_Blocker();
?>
