<?php
/*
Plugin Name: WAO Multisite Email Blocker
Description: Bloquea correos electrónicos o dominios específicos en el checkout de WooCommerce en toda la red.
Network: true
Version: 1.6
Author: Maykel-J
*/

if (!defined('ABSPATH')) exit;

class Multisite_Email_Blocker {

    const OPTION_KEY = 'blocked_emails_list';

    public function __construct() {
        if (!is_multisite()) return;
		add_action('admin_enqueue_scripts', [$this, 'enqueue_admin_styles']);
        add_action('network_admin_menu', [$this, 'add_network_menu']);
        add_action('network_admin_edit_save_blocked_emails', [$this, 'save_blocked_emails']);
        add_action('network_admin_edit_save_blocked_languages', [$this, 'save_blocked_languages']);

        add_action('woocommerce_checkout_process', [$this, 'validate_checkout_email']);
        add_action('wp_enqueue_scripts', [$this, 'enqueue_checkout_script']);
		
    }

    public function add_network_menu() {
        add_menu_page(
            'Email Blocker',
            'Email Blocker',
            'manage_network_options',
            'email-blocker',
            [$this, 'render_settings_page'],
            'dashicons-email-alt2',
            30
        );
    }

    public function render_settings_page() {
        $active_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'emails';
        $emails = get_site_option(self::OPTION_KEY, '');
        $languages = get_site_option('wao_blocked_email_languages', [
            'en_US' => 'This email address is blocked.',
            'de_DE' => 'Diese E-Mail-Adresse ist gesperrt.',
            'es_ES' => 'Esta dirección de correo está bloqueada.',
            'fr_FR' => 'Cette adresse e-mail est bloquée.',
            'it_IT' => 'Questo indirizzo email è bloccato.',
        ]);
        ?>
        <div class="wrap">
            <h1>Email Blocker Settings</h1>
            <h2 class="nav-tab-wrapper">
                <a href="?page=email-blocker&tab=emails" class="nav-tab <?php echo ($active_tab === 'emails') ? 'nav-tab-active' : ''; ?>">Blocked Emails</a>
                <a href="?page=email-blocker&tab=languages" class="nav-tab <?php echo ($active_tab === 'languages') ? 'nav-tab-active' : ''; ?>">Languages</a>
            </h2>

            <?php if ($active_tab === 'emails') : ?>
                <form method="post" action="edit.php?action=save_blocked_emails">
                    <?php wp_nonce_field('save_blocked_emails_nonce'); ?>
                    <textarea name="blocked_emails" rows="10" cols="80" placeholder="email1@example.com, @domain.com"><?php echo esc_textarea($emails); ?></textarea>
                    <p class="description">Enter blocked emails or domains separated by comma.<br>
                    Examples: <code>spam@example.com</code>, <code>@baddomain.com</code></p>
                    <?php submit_button('Save'); ?>
                </form>
            <?php elseif ($active_tab === 'languages') : ?>
                <form method="post" action="edit.php?action=save_blocked_languages">
                    <?php wp_nonce_field('save_blocked_languages_nonce'); ?>
                    <table class="form-table">
                        <?php foreach ($languages as $locale => $text) : ?>
                            <tr>
                                <th scope="row"><?php echo esc_html($locale); ?></th>
                                <td><input type="text" name="blocked_langs[<?php echo esc_attr($locale); ?>]" value="<?php echo esc_attr($text); ?>" style="width: 100%;"></td>
                            </tr>
                        <?php endforeach; ?>
                    </table>
                    <?php submit_button('Save Messages'); ?>
                </form>
            <?php endif; ?>
        </div>
        <?php
    }

    public function save_blocked_emails() {
        check_admin_referer('save_blocked_emails_nonce');
        if (!current_user_can('manage_network_options')) {
            wp_die(__('You do not have permission to update this setting.'));
        }

        $raw_input = isset($_POST['blocked_emails']) ? sanitize_text_field($_POST['blocked_emails']) : '';
        update_site_option(self::OPTION_KEY, $raw_input);

        wp_redirect(network_admin_url('admin.php?page=email-blocker&tab=emails&updated=1'));
        exit;
    }

    public function save_blocked_languages() {
        check_admin_referer('save_blocked_languages_nonce');
        if (!current_user_can('manage_network_options')) {
            wp_die(__('You do not have permission to update this setting.'));
        }

        $langs = isset($_POST['blocked_langs']) ? (array) $_POST['blocked_langs'] : [];
        $langs = array_map('sanitize_text_field', $langs);
        update_site_option('wao_blocked_email_languages', $langs);

        wp_redirect(network_admin_url('admin.php?page=email-blocker&tab=languages&updated=1'));
        exit;
    }

    public function get_blocked_items() {
        $raw = get_site_option(self::OPTION_KEY, '');
        $list = array_map('trim', explode(',', $raw));
        return array_filter($list);
    }

    public function validate_checkout_email() {
        if (!function_exists('WC')) return;

        $email = isset($_POST['billing_email']) ? sanitize_email($_POST['billing_email']) : '';
        $blocked = $this->get_blocked_items();

        foreach ($blocked as $entry) {
            if (stripos($email, $entry) !== false) {
                $langs = get_site_option('wao_blocked_email_languages', []);
                $locale = get_option('WPLANG');
                if (empty($locale)) {
                    $locale = 'en_US';
                }
                $message = isset($langs[$locale]) ? $langs[$locale] : ($langs['en_US'] ?? 'This email address is blocked.');
                wc_add_notice($message, 'error');
                break;
            }
        }
    }

    public function enqueue_checkout_script() {
        if (!function_exists('is_checkout')) return;
        if (!is_checkout()) return;

        $blocked = $this->get_blocked_items();
        if (empty($blocked)) return;

        $blocked_json = wp_json_encode($blocked);

        $langs = get_site_option('wao_blocked_email_languages', []);
        $locale = get_option('WPLANG');
        if (empty($locale)) {
            $locale = 'en_US';
        }
        $locale = str_replace('-', '_', $locale);
        $message = isset($langs[$locale]) ? $langs[$locale] : ($langs['en_US'] ?? 'This email address is blocked.');
        $message_js = esc_js($message);

        $js = <<<JS
jQuery(document).ready(function($) {
    let blocked = $blocked_json;
    let message = "$message_js";
    let \$email = $("#billing_email");
    let \$form = $("form.checkout");

    function checkBlockedEmail(email) {
        email = email.toLowerCase();
        for (let i = 0; i < blocked.length; i++) {
            if (email.includes(blocked[i].toLowerCase())) {
                return true;
            }
        }
        return false;
    }

    function showError() {
        if (!$(".email-blocked-error").length) {
            \$email.after("<div class='woocommerce-error email-blocked-error'>" + message + "</div>");
        }
    }

    function clearError() {
        $(".email-blocked-error").remove();
    }

    \$email.on("input", function() {
        let val = $(this).val();
        clearError();
        if (checkBlockedEmail(val)) {
            showError();
        }
    });

    \$form.on("checkout_place_order", function(e) {
        let val = \$email.val();
        clearError();
        if (checkBlockedEmail(val)) {
            showError();
            return false;
        }
        return true;
    });
});
JS;

        wp_add_inline_script('jquery', $js);
    }
	
public function enqueue_admin_styles($hook) {
    // Solo cargar en el Network Admin para el plugin
    if (!is_network_admin() || $hook !== 'toplevel_page_email-blocker') {
        return;
    }

    echo '<style>
        .wrap h1 {
            font-size: 24px;
            margin-bottom: 20px;
        }
        .nav-tab-wrapper {
            margin-bottom: 20px;
        }
        textarea[name="blocked_emails"] {
            font-family: monospace;
            border: 1px solid #ccd0d4;
            border-radius: 6px;
            padding: 10px;
            width: 100%;
            max-width: 800px;
            min-height: 200px;
            background: #fff;
            box-shadow: inset 0 1px 3px rgba(0,0,0,0.05);
			margin-top: 10px;
        }
        .form-table input[type="text"] {
            border-radius: 4px;
            padding: 6px 10px;
            border: 1px solid #ccd0d4;
            width: 100%;
            max-width: 600px;
        }
        .form-table th {
            width: 100px;
            text-align: left;
            font-weight: 600;
            vertical-align: top;
            padding-top: 10px;
        }
        .form-table td {
            padding-top: 10px;
        }
        .description {
            margin-top: 10px;
            color: #666;
        }
    </style>';
}

}


new Multisite_Email_Blocker();
