<?php
/*
Plugin Name: WAO Plugin Network
Description: Gestiona plugins en todos los sitios del multisitio desde un solo lugar con AJAX (acordeón, filtros, lazy-load).
Version: 2.0.3
Author: Maykel J. Menendez
License: GPLv2 or later
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: wao-plugin-network
*/

if (!defined('ABSPATH')) exit;

add_action('network_admin_menu', function () {
    add_menu_page(
        __('Gestión de Plugins por Sitio', 'wao-plugin-network'),
        __('Gestión de Plugin', 'wao-plugin-network'),
        'manage_network_plugins',
        'wao-plugin-network',
        'wao_pnm_render_admin_page',
        'dashicons-admin-plugins',
        70
    );
});

function wao_pnm_render_admin_page() {
    if (!current_user_can('manage_network_plugins')) {
        wp_die(__('No tienes permisos suficientes.', 'wao-plugin-network'));
    }

    require_once ABSPATH . 'wp-admin/includes/plugin.php';

    $all_plugins = get_plugins();
    $sites       = get_sites(['number' => 10000, 'fields' => 'ids']); // ids para rendimiento
    $network_active_plugins = array_keys(get_site_option('active_sitewide_plugins', []));
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Gestión de Plugins Multisitio', 'wao-plugin-network'); ?></h1>

        <div class="pnm-toolbar">
            <input type="text" id="pnm-filter" placeholder="<?php esc_attr_e('Filtrar plugins...', 'wao-plugin-network');?>"/>
            <div class="pnm-chips">
                <button class="button button-small pnm-chip button-primary" data-filter="all"><?php esc_html_e('Todos','wao-plugin-network');?></button>
                <button class="button button-small pnm-chip" data-filter="network_active"><?php esc_html_e('Activos en red','wao-plugin-network');?></button>
                <button class="button button-small pnm-chip" data-filter="network_inactive"><?php esc_html_e('No activos en red','wao-plugin-network');?></button>
            </div>
        </div>

        <div id="pnm-container">
            <?php foreach ($all_plugins as $plugin_path => $plugin_data): ?>
                <?php 
                    $is_network_active = in_array($plugin_path, $network_active_plugins, true);
                    $network_status = $is_network_active ? '🟢 ' . __('Activo en red','wao-plugin-network') : '🔴 ' . __('No activo en red','wao-plugin-network');
                    list($count_active, $count_inactive) = wao_pnm_get_counts($plugin_path, $sites);
                ?>
                <div class="pnm-plugin" data-plugin="<?php echo esc_attr($plugin_path); ?>">
                    <button class="pnm-accordion-toggle" data-plugin="<?php echo esc_attr($plugin_path); ?>" aria-expanded="false">
                        <span class="pnm-plugin-name"><?php echo esc_html($plugin_data['Name']); ?></span>
                        <span class="pnm-badges">
                            <span class="pnm-badge pnm-badge-active" data-count="<?php echo (int)$count_active; ?>"><?php printf(__('Activos: %d','wao-plugin-network'), (int)$count_active); ?></span>
                            <span class="pnm-badge pnm-badge-inactive" data-count="<?php echo (int)$count_inactive; ?>"><?php printf(__('Inactivos: %d','wao-plugin-network'), (int)$count_inactive); ?></span>
                        </span>
                        <span class="pnm-network-indicator"><?php echo esc_html($network_status); ?></span>
                    </button>
                    <div class="pnm-accordion-content" data-loaded="0" style="display:none;">
                        <div class="pnm-network-row">
                            <strong><?php esc_html_e('Estado global:','wao-plugin-network'); ?></strong>
                            <span class="pnm-status"><?php echo esc_html($network_status); ?></span>
                            <?php
                                $network_action = $is_network_active ? 'network_deactivate' : 'network_activate';
                                $network_label  = $is_network_active ? __('Desactivar red','wao-plugin-network') : __('Activar red','wao-plugin-network');
                                $network_class  = $is_network_active ? 'button-secondary' : 'button-primary';
                            ?>
                            <button class="pnm-toggle-network button <?php echo esc_attr($network_class); ?>" data-action="<?php echo esc_attr($network_action); ?>">
                                <?php echo esc_html($network_label); ?>
                            </button>
                        </div>

                        <div class="pnm-sites">
                            <?php
                                // Renderizamos estructura de filas vacía (el estado se cargará lazy por AJAX)
                                $half = ceil(count($sites) / 2);
                                $chunks = array_chunk($sites, $half);
                                foreach ($chunks as $column_sites): ?>
                                    <div class="pnm-site-column">
                                    <?php foreach ($column_sites as $sid): 
                                        $blog = get_blog_details($sid);
                                        if (!$blog) continue;
                                    ?>
                                        <div class="pnm-site-row" data-site="<?php echo esc_attr($sid); ?>">
                                            <?php $plugins_url = get_admin_url($sid, 'plugins.php'); ?>
<a href="<?php echo esc_url($plugins_url); ?>" target="_blank" rel="noopener">
    <?php echo esc_html($blog->blogname); ?> (ID: <?php echo (int)$sid; ?>)
</a>
                                            <span class="pnm-status">—</span>
                                            <button class="pnm-toggle-btn button" data-action="activate" disabled><?php esc_html_e('Activar','wao-plugin-network'); ?></button>
                                        </div>
                                    <?php endforeach; ?>
                                    </div>
                                <?php endforeach; ?>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
    <?php
}

add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook !== 'toplevel_page_wao-plugin-network') return;
    wp_enqueue_style('pnm-style', plugin_dir_url(__FILE__) . 'pnm.css', [], '2.0.3');
    wp_enqueue_script('pnm-ajax-script', plugin_dir_url(__FILE__) . 'pnm.js', ['jquery'], '2.0.3', true);
    wp_localize_script('pnm-ajax-script', 'pnm_ajax', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce'    => wp_create_nonce('pnm_ajax_nonce')
    ]);
});

// Toggle de plugins (individual y red)
add_action('wp_ajax_pnm_toggle_plugin', function () {
    if (!current_user_can('manage_network_plugins') || !check_ajax_referer('pnm_ajax_nonce', 'nonce', false)) {
        wp_send_json_error('No autorizado');
    }

    require_once ABSPATH . 'wp-admin/includes/plugin.php';

    $plugin = sanitize_text_field($_POST['plugin'] ?? '');
    $site_id = intval($_POST['site_id'] ?? 0);
    $action  = sanitize_text_field($_POST['action_type'] ?? '');
    $status  = '';

    if (!$plugin || !in_array($action, ['activate','deactivate','network_activate','network_deactivate'], true)) {
        wp_send_json_error('Parámetros incorrectos');
    }

    $all = array_keys(get_plugins());
    if (!in_array($plugin, $all, true)) {
        wp_send_json_error('Plugin inválido');
    }

    if ($action === 'network_activate') {
        activate_plugin($plugin, '', true);
        $status = 'Activo';
    } elseif ($action === 'network_deactivate') {
        deactivate_plugins($plugin, true);
        $status = 'Inactivo';
    } elseif ($site_id) {
        switch_to_blog($site_id);
        if ($action === 'activate') {
            $result = activate_plugin($plugin);
            $status = 'Activo';
        } else {
            deactivate_plugins($plugin, false, false);
            $result = true;
            $status = 'Inactivo';
        }
        restore_current_blog();
        if (isset($result) && is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
    }

    wao_pnm_bust_counts_cache($plugin);
    wp_send_json_success(['status' => $status]);
});

// Lazy-load de estados por sitio para un plugin
add_action('wp_ajax_wao_get_plugin_states', function () {
    if (!current_user_can('manage_network_plugins') || !check_ajax_referer('pnm_ajax_nonce', 'nonce', false)) {
        wp_send_json_error('No autorizado');
    }
    require_once ABSPATH . 'wp-admin/includes/plugin.php';

    $plugin = sanitize_text_field($_GET['plugin'] ?? '');
    if (!$plugin) wp_send_json_error('Plugin faltante');

    $all = array_keys(get_plugins());
    if (!in_array($plugin, $all, true)) {
        wp_send_json_error('Plugin inválido');
    }

    $sites = get_sites(['fields' => 'ids', 'number' => 10000]);
    $data = [];
    $active_count = 0; $inactive_count = 0;
    foreach ($sites as $sid) {
        switch_to_blog($sid);
        $active_plugins = get_option('active_plugins', []);
        $is_active = in_array($plugin, $active_plugins, true);
        $data[$sid] = $is_active ? 'Activo' : 'Inactivo';
        $is_active ? $active_count++ : $inactive_count++;
        restore_current_blog();
    }
    wp_send_json_success([
        'states' => $data,
        'counts' => ['active' => $active_count, 'inactive' => $inactive_count],
    ]);
});


// Cuenta sitios activos/inactivos con caché (10 min)
function wao_pnm_get_counts($plugin, $sites_ids) {
    $key = 'wao_pnm_counts_' . md5($plugin);
    $cached = get_site_transient($key);
    if (is_array($cached) && isset($cached['active'], $cached['inactive'])) {
        return [$cached['active'], $cached['inactive']];
    }
    $active = 0; $inactive = 0;
    foreach ($sites_ids as $sid) {
        switch_to_blog($sid);
        $active_plugins = get_option('active_plugins', []);
        $is_active = in_array($plugin, $active_plugins, true);
        $is_active ? $active++ : $inactive++;
        restore_current_blog();
    }
    set_site_transient($key, ['active' => $active, 'inactive' => $inactive], 10 * MINUTE_IN_SECONDS);
    return [$active, $inactive];
}
// Invalidar caché cuando haya cambios
function wao_pnm_bust_counts_cache($plugin) {
    delete_site_transient('wao_pnm_counts_' . md5($plugin));
}

