
<?php
/**
 * Plugin Name: WAOLM Network CPT Manager (Con Nuevo y Filtros)
 * Description: Registra y administra el CPT "licensemanagerwao" en Network Admin con listado, edición, bulk actions, búsqueda, botón "Add New Key" y filtros por producto y estado.
 * Version: 2.0 
 * Change log: Manejo de custompost type licensemanagerwao en network admin
 * Fecha actualizacion: 24 de febrero de 2025
 * Author: Arturo 
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * REGISTRO DEL CPT EN NETWORK ADMIN
 */
function waolm_register_network_cpt() {
	if ( ! is_network_admin() ) {
		return;
	}
	
	$labels = array(
		'name'                  => 'WAO License Manager',
		'singular_name'         => 'Key',
		'all_items'             => 'All Keys',
		'add_new'               => 'Add new Key',
		'add_new_item'          => 'Add new Key',
		'edit_item'             => 'Edit Key',
		'new_item'              => 'New Key',
		'view_item'             => 'View Key',
		'search_items'          => 'Search Keys',
		'not_found'             => 'Keys not found',
		'not_found_in_trash'    => 'Keys not found in trash'
	);
	
	$args = array(
		'public'          => true,
		'labels'          => $labels,
		'menu_position'   => 12,
		'capability_type' => 'post',
		'menu_icon'       => 'dashicons-products',
		'supports'        => array( 'title' ),
		// No se muestra el menú por defecto ya que usaremos el menú personalizado.
		'show_in_menu'    => false,
	);
	
	register_post_type( 'licensemanagerwao', $args );
}
//add_action( 'init', 'waolm_register_network_cpt' );  

/**
 * MENÚ EN NETWORK ADMIN
 */
function waolm_network_menu() {
	if ( ! is_network_admin() ) {
		return;
	}
	
	// Menú principal
	add_menu_page(
		'WAOLM License Manager',         // Título de la página
		'WAOLM License Manager',         // Título del menú
		'manage_network',                // Capacidad requerida
		'waolm-network-cpt',             // Slug del menú
		'waolm_network_cpt_page',        // Callback de la página (listado)
		'dashicons-admin-network',            // Ícono
		6                                // Posición
	);
	
	// Submenú "ABM TABLE"
	add_submenu_page(
		'waolm-network-cpt',             // Parent slug
        'Gestión de Licencias',
        'Licencias WAOLM',
        'manage_network',
        'waolm_licensemanager_admin',
        'waolm_render_licensemanager_admin_page'
	);

	// Submenú "Add New Key"
	add_submenu_page(
		'waolm-network-cpt',             // Parent slug
		'Add New Key',                   // Título de la página
		'Add New Key',                   // Título del menú
		'manage_network',                // Capacidad requerida
		'waolm-network-cpt-new',         // Slug
		'waolm_network_new_page'         // Callback para la nueva key
	);

	// Submenú "Server Settings"
    add_submenu_page(
        'waolm-network-cpt',  // slug del menú principal
        'Server settings',  // Título de la página
        'Server Settings',  // Título del menú
        'manage_network', // Capacidad necesaria para ver este menú
        'waolm_server_config',   // Slug del submenú
        'waolm_server_config_page' // Función que renderiza la página del submenú
    );


}

add_action( 'network_admin_menu', 'waolm_network_menu' );

/**
 * Asegurarse de que la clase WP_List_Table esté cargada.
 */
if ( ! class_exists( 'WP_List_Table' ) ) {
	require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

/**
 * CLASE PERSONALIZADA WP_List_Table PARA WAOLM
 */
class WAOLM_List_Table extends WP_List_Table {
	
	public function __construct() {
		parent::__construct( array(
			'singular' => 'licensemanagerwao',
			'plural'   => 'licensemanagerwaos',
			'ajax'     => false,
		) );
	}
	
	// Se definen las columnas sin la columna "ID"
	public function get_columns() {
		$columns = array(
			'cb'          => '<input type="checkbox" />',
			'title'       => 'Title',
			'product'     => 'Product',
			'activations' => 'Activations',
			'dispatched'  => 'Dispatched',
			'status'      => 'Status',
			'priority'    => 'Priority',
			'sku'         => 'SKU',
		);
		return $columns;
	}
	
	public function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="post[]" value="%s" />',
			$item->ID
		);
	}
	
	// Columna "title" con enlaces de edición y borrado.
	public function column_title( $item ) {
		$edit_link = add_query_arg( array(
			'page'   => 'waolm-network-cpt',
			'action' => 'edit',
			'post'   => $item->ID,
		), network_admin_url( 'admin.php' ) );
		
		$title = sprintf( '<strong><a href="%s">%s</a></strong>', esc_url( $edit_link ), $item->post_title );
		
		$actions = array(
			'edit'   => sprintf( '<a href="%s">Edit</a>', esc_url( $edit_link ) ),
			'delete' => sprintf( '<a href="%s">Delete</a>', 
				wp_nonce_url(
					add_query_arg( array(
						'page'   => 'waolm-network-cpt',
						'action' => 'delete',
						'post'   => $item->ID,
					), network_admin_url( 'admin.php' ) ), 'delete_post_' . $item->ID 
				)
			),
		);
		
		return $title . $this->row_actions( $actions );
	}
	
	// En la columna "product" se muestra el nombre del producto (almacenado en _product_name)
	public function column_product( $item ) {
		return get_post_meta( $item->ID, '_product_name', true );
	}
	
	public function column_activations( $item ) {
		return get_post_meta( $item->ID, '_activations', true );
	}
	
	public function column_dispatched( $item ) {
		return get_post_meta( $item->ID, '_dispatched', true );
	}
	
	public function column_status( $item ) {
		return $item->post_status == 'publish' ? 'Active' : 'Inactive';
	}
	
	public function column_priority( $item ) {
		return $item->menu_order;
	}
	
	public function column_sku( $item ) {
		return get_post_meta( $item->ID, '_waolm_sku', true );
	}
	
	public function get_sortable_columns() {
		$sortable_columns = array(
			'title'    => array( 'title', false ),
			'priority' => array( 'priority', false ),
		);
		return $sortable_columns;
	}
	
	// Se incorpora el Search Box y los filtros (por producto y estado)
	public function prepare_items() {
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();
		$this->_column_headers = array( $columns, $hidden, $sortable );
		
		$per_page = 20;
		$current_page = $this->get_pagenum();
		
		$args = array(
			'post_type'      => 'licensemanagerwao',
			'posts_per_page' => $per_page,
			'paged'          => $current_page,
		);
		
		// Filtro por producto (buscando en _product_name)
		if ( isset( $_REQUEST['product_filter'] ) && $_REQUEST['product_filter'] != '' ) {
			$args['meta_query'][] = array(
				'key'     => '_product_name',
				'value'   => sanitize_text_field( $_REQUEST['product_filter'] ),
				'compare' => 'LIKE'
			);
		}
		
		// Filtro por estado (Active/Inactive)
		if ( isset( $_REQUEST['status_filter'] ) && $_REQUEST['status_filter'] != '' ) {
			$status = sanitize_text_field( $_REQUEST['status_filter'] );
			$args['post_status'] = ( $status === 'Active' ) ? 'publish' : 'draft';
		}
		
		// Búsqueda en título
		if ( isset( $_REQUEST['s'] ) && ! empty( $_REQUEST['s'] ) ) {
			$args['s'] = sanitize_text_field( $_REQUEST['s'] );
		}
		
		if ( ! empty( $_REQUEST['orderby'] ) ) {
			$args['orderby'] = sanitize_text_field( $_REQUEST['orderby'] );
		}
		if ( ! empty( $_REQUEST['order'] ) ) {
			$args['order'] = sanitize_text_field( $_REQUEST['order'] );
		}
		
		$query = new WP_Query( $args );
		$this->items = $query->posts;
		
		$total_items = $query->found_posts;
		$this->set_pagination_args( array(
			'total_items' => $total_items,
			'per_page'    => $per_page,
		) );
	}
	
	// Definimos las acciones masivas (Bulk Actions)
	public function get_bulk_actions() {
		$actions = array(
			'bulk-delete' => 'Delete'
		);
		return $actions;
	}
	
	// Procesamos la acción masiva
	public function process_bulk_action() {
		if ( 'bulk-delete' === $this->current_action() ) {
			if ( isset( $_REQUEST['post'] ) && is_array( $_REQUEST['post'] ) ) {
				foreach( $_REQUEST['post'] as $post_id ) {
					$post_id = absint( $post_id );
					wp_delete_post( $post_id, true );
				}
			}
		}
	}
	
	public function extra_tablenav( $which ) {
		if ( $which == "top" ) { ?>
			<div class="alignleft actions">
				<label for="product_filter" class="screen-reader-text">Filtrar por Producto</label>
				<select name="product_filter" id="product_filter">
					<option value="">Todos los Productos</option>
					<?php
					global $wpdb;
					$results = $wpdb->get_results( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key = '_product_name' AND meta_value <> '' ORDER BY meta_value" );
					foreach ( $results as $row ) {
						$selected = ( isset($_REQUEST['product_filter']) && $_REQUEST['product_filter'] == $row->meta_value ) ? ' selected="selected"' : '';
						echo '<option value="' . esc_attr( $row->meta_value ) . '"' . $selected . '>' . esc_html( $row->meta_value ) . '</option>';
					}
					?>
				</select>

				<label for="status_filter" class="screen-reader-text">Filtrar por Estado</label>
				<select name="status_filter" id="status_filter">
					<option value="">Todos los Estados</option>
					<option value="Active" <?php selected( (isset($_REQUEST['status_filter']) ? $_REQUEST['status_filter'] : ''), 'Active' ); ?>>Active</option>
					<option value="Inactive" <?php selected( (isset($_REQUEST['status_filter']) ? $_REQUEST['status_filter'] : ''), 'Inactive' ); ?>>Inactive</option>
				</select>

				<?php submit_button( 'Filtrar', 'secondary', '', false ); ?>
			</div>
		<?php }
	}
	
}

/**
 * PÁGINA PRINCIPAL: LISTADO Y ACCIONES
 */
function waolm_network_cpt_page() {
	// Procesa la eliminación individual (vía GET)
	if ( isset( $_GET['action'] ) && $_GET['action'] === 'delete' && ! empty( $_GET['post'] ) ) {
		$post_id = absint( $_GET['post'] );
		if ( wp_verify_nonce( $_GET['_wpnonce'], 'delete_post_' . $post_id ) ) {
			wp_delete_post( $post_id, true );
			echo '<div class="updated notice"><p>Post deleted.</p></div>';
		}
	}
	
	// Si se solicita la edición, cargar la vista de edición
	if ( isset( $_GET['action'] ) && $_GET['action'] === 'edit' && ! empty( $_GET['post'] ) ) {
		waolm_network_edit_page( absint( $_GET['post'] ) );
		return;
	}
	
	$list_table = new WAOLM_List_Table();
	$list_table->process_bulk_action();
	$list_table->prepare_items();
	?>
	<div class="wrap">
		<h1>WAOLM License Manager
		<!-- Botón "Add New Key" -->
		<span>
			<a href="<?php echo network_admin_url('admin.php?page=waolm-network-cpt-new'); ?>" class="page-title-action">Add New Key</a>
		</span>
		</h1>
		
		
		<!-- Filtro por Producto, Estado y Search Box -->
		<form method="get">
			<input type="hidden" name="page" value="<?php echo esc_attr( $_REQUEST['page'] ); ?>">
			<?php 
			$list_table->search_box('Search Keys', 'waolm_search'); 
			?>
		</form>
		
		<form method="get">
			<input type="hidden" name="page" value="<?php echo esc_attr( $_REQUEST['page'] ); ?>">
			<?php $list_table->display(); ?>
		</form>
	</div>
	<?php
}

/**
 * PÁGINA DE EDICIÓN DEL REGISTRO
 */
function waolm_network_edit_page( $post_id ) {
	$post = get_post( $post_id );
	if ( ! $post || $post->post_type !== 'licensemanagerwao' ) {
		echo '<div class="error"><p>Registro no encontrado.</p></div>';
		return;
	}
	
	// Procesar envío del formulario
	if ( isset( $_POST['waolm_nonce'] ) && wp_verify_nonce( $_POST['waolm_nonce'], 'waolm_edit_' . $post_id ) ) {
		$updated_post = array(
			'ID'         => $post_id,
			'post_title' => sanitize_text_field( $_POST['title'] ),
			'post_status'=> ( 'Active' === sanitize_text_field( $_POST['status'] ) ) ? 'publish' : 'draft',
			'menu_order' => intval( $_POST['priority'] ),
		);
		wp_update_post( $updated_post );
		
		// Actualizar metadatos
		$product = sanitize_text_field( $_POST['product'] );
		update_post_meta( $post_id, '_product', $product );
		$sku = get_post_meta( $product, '_sku', true );
		update_post_meta( $post_id, '_waolm_sku', $sku );
		$product_obj = get_post( $product );
		if ( $product_obj ) {
			update_post_meta( $post_id, '_product_name', $product_obj->post_title );
		}
		$activations = sanitize_text_field( $_POST['activations'] );
		update_post_meta( $post_id, '_activations', $activations );
		$dispatched = sanitize_text_field( $_POST['dispatched'] );
		update_post_meta( $post_id, '_dispatched', $dispatched );
		$priority = sanitize_text_field( $_POST['priority'] );
		update_post_meta( $post_id, '_priority', $priority );
		
		echo '<div class="updated notice"><p>Registro actualizado.</p></div>';
		$post = get_post( $post_id );
	}
	
	?>
	<div class="wrap postbox">
		<h1 class="postbox-header"  style="padding:10px;">Edit Key: ID <?php echo $post_id; ?></h1>
		<form method="post" style="padding:10px;">
			<?php wp_nonce_field( 'waolm_edit_' . $post_id, 'waolm_nonce' ); ?>
			<table class="form-table">
				<tr>
					<th><label for="title">KEY</label></th>
					<td><input type="text" name="title" id="title" value="<?php echo esc_attr( $post->post_title ); ?>" class="regular-text"></td>
				</tr>
				<tr>
					<th><label for="product">Product</label></th>
					<td>
						<select name="product" id="product">
							<?php
							$args = array(
								'numberposts' => -1,
								'post_type'   => 'product',
								'post_status' => 'publish',
								'orderby'     => 'post_title',
								'order'       => 'ASC'
							);
							$productos = get_posts( $args );
							echo '<option value="">Select a product</option>';
							foreach ( $productos as $prod ) {
								$selected = ( $prod->ID == get_post_meta( $post_id, '_product', true ) ) ? 'selected="selected"' : '';
								echo '<option value="' . esc_attr( $prod->ID ) . '" ' . $selected . '>' . esc_html( $prod->post_title ) . '</option>';
							}
							?>
						</select>
					</td>
				</tr>
				<tr>
					<th><label for="activations">Activations</label></th>
					<td><input type="number" name="activations" id="activations" value="<?php echo esc_attr( get_post_meta( $post_id, '_activations', true ) ); ?>"></td>
				</tr>
				<tr>
					<th><label for="dispatched">Dispatched</label></th>
					<td><input type="number" name="dispatched" id="dispatched" value="<?php echo esc_attr( get_post_meta( $post_id, '_dispatched', true ) ); ?>"></td>
				</tr>
				<tr>
					<th><label for="priority">Priority</label></th>
					<td><input type="number" name="priority" id="priority" value="<?php echo esc_attr( get_post_meta( $post_id, '_priority', true ) ); ?>"></td>
				</tr>
				<tr>
					<th><label for="status">Status</label></th>
					<td>
						<?php $status = $post->post_status == 'publish' ? 'Active' : 'Inactive'; ?>
						<select name="status" id="status">
							<option value="Active" <?php selected( $status, 'Active' ); ?>>Active</option>
							<option value="Inactive" <?php selected( $status, 'Inactive' ); ?>>Inactive</option>
						</select>
					</td>
				</tr>
			</table>
			<?php submit_button( 'Save changes' ); ?>
		</form>
		<p style="padding-left:10px;margin-top:-20px;"><span class="dashicons dashicons-arrow-left-alt"></span><a href="<?php echo network_admin_url('admin.php?page=waolm-network-cpt'); ?>">Volver a la lista</a></p>
	</div>
	<?php
}

/**
 * PÁGINA "ADD NEW KEY": CREACIÓN DE UN NUEVO REGISTRO
 */
function waolm_network_new_page() {
	// Procesar envío del formulario para crear la nueva key
	if ( isset( $_POST['waolm_nonce'] ) && wp_verify_nonce( $_POST['waolm_nonce'], 'waolm_new' ) ) {
		$new_post = array(
			'post_title'   => sanitize_text_field( $_POST['title'] ),
			'post_status'  => ( 'Active' === sanitize_text_field( $_POST['status'] ) ) ? 'publish' : 'draft',
			'post_type'    => 'licensemanagerwao',
			'menu_order'   => intval( $_POST['priority'] ),
		);
		$post_id = wp_insert_post( $new_post );
		
		if ( $post_id && ! is_wp_error( $post_id ) ) {
			// Guardar metadatos
			$product = sanitize_text_field( $_POST['product'] );
			update_post_meta( $post_id, '_product', $product );
			$sku = get_post_meta( $product, '_sku', true );
			update_post_meta( $post_id, '_waolm_sku', $sku );
			$product_obj = get_post( $product );
			if ( $product_obj ) {
				update_post_meta( $post_id, '_product_name', $product_obj->post_title );
			}
			$activations = sanitize_text_field( $_POST['activations'] );
			update_post_meta( $post_id, '_activations', $activations );
			$dispatched = sanitize_text_field( $_POST['dispatched'] );
			update_post_meta( $post_id, '_dispatched', $dispatched );
			$priority = sanitize_text_field( $_POST['priority'] );
			update_post_meta( $post_id, '_priority', $priority );
			
			echo '<div class="updated notice"><p>Nueva Key creada.</p></div>';
			// Redirigir al listado o a la página de edición según convenga
		} else {
			echo '<div class="error"><p>Error al crear la nueva Key.</p></div>';
		}
	}
	?>
	<div class="wrap postbox">
		<h1 class="postbox-header" style="padding:10px;">Add New Key:</h1>
		<form method="post" style="padding:10px;">
			<?php wp_nonce_field( 'waolm_new', 'waolm_nonce' ); ?>
			<table class="form-table">
				<tr>
					<th><label for="title">KEY</label></th>
					<td><input type="text" name="title" id="title" value="" class="regular-text"></td>
				</tr>
				<tr>
					<th><label for="product">Product</label></th>
					<td>
						<select name="product" id="product">
							<?php
							$args = array(
								'numberposts' => -1,
								'post_type'   => 'product',
								'post_status' => 'publish',
								'orderby'     => 'post_title',
								'order'       => 'ASC'
							);
							$productos = get_posts( $args );
							echo '<option value="">Select a product</option>';
							foreach ( $productos as $prod ) {
								echo '<option value="' . esc_attr( $prod->ID ) . '">' . esc_html( $prod->post_title ) . '</option>';
							}
							?>
						</select>
					</td>
				</tr>
				<tr>
					<th><label for="activations">Activations</label></th>
					<td><input type="number" name="activations" id="activations" value="0"></td>
				</tr>
				<tr>
					<th><label for="dispatched">Dispatched</label></th>
					<td><input type="number" name="dispatched" id="dispatched" value="0"></td>
				</tr>
				<tr>
					<th><label for="priority">Priority</label></th>
					<td><input type="number" name="priority" id="priority" value="0"></td>
				</tr>
				<tr>
					<th><label for="status">Status</label></th>
					<td>
						<select name="status" id="status">
							<option value="Active">Active</option>
							<option value="Inactive">Inactive</option>
						</select>
					</td>
				</tr>
			</table>
			<?php submit_button( 'Save New Key' ); ?>
		</form>
		<p style="padding-left:10px;margin-top:-20px;"><span class="dashicons dashicons-arrow-left-alt"></span><a href="<?php echo network_admin_url('admin.php?page=waolm-network-cpt'); ?>">Volver a la lista</a></p>
		
	</div>
	<?php
}


// PAGINA DE CONFIGURACION DEL SERVER

function waolm_server_config_page() {
    $active_tab = isset($_GET['tab']) ? $_GET['tab'] : 'server_config';
    waolm_save_network_options();
	?>
    <div class="wrap">
        <h2>Configuración WAOLM License Manager</h2>
        <h2 class="nav-tab-wrapper">
            <a href="?page=waolm_server_config&tab=server_config" class="nav-tab <?php echo $active_tab == 'server_config' ? 'nav-tab-active' : ''; ?>">Server Configuración</a>
            <a href="?page=waolm_server_config&tab=api" class="nav-tab <?php echo $active_tab == 'api' ? 'nav-tab-active' : ''; ?>">API</a>
        </h2>

        <?php if ($active_tab == 'server_config') : ?>
            <p>Aun nada que configurar.</p>
        <?php else : ?>
            <form method="post" action="">
                <?php
                wp_nonce_field('waolm_update_api_key');
				$waolm_server_options = get_blog_option(1,'_waolm_server_options',[]);
				$waolm_server_api_key=$waolm_server_options['waolm_server_api_key']; 
                ?>
                <h3>API</h3>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="waolm_server_api_key">API Server Access Key</label></th>
                        <td>
                            <input type="text" id="waolm_server_api_key" name="waolm_server_api_key" value="<?php echo $waolm_server_api_key; ?>" class="regular-text">
                            <button type="button" onclick="generateApiKey()">Generar</button>
                        </td>
                    </tr>
                </table>
                <?php submit_button('Guardar Cambios', 'primary', 'submit', true); ?>
            </form>
            <script>
            function generateApiKey() {
                var result = '';
                var characters = 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789';
                var charactersLength = characters.length;
                for (var i = 0; i < 15; i++) {
                    result += characters.charAt(Math.floor(Math.random() * charactersLength));
                }
                document.getElementById('waolm_server_api_key').value = result;
            }
            </script>
        <?php endif; ?>
    </div>
    <?php
}


function waolm_save_network_options() {
    if (isset($_POST['submit'])) {
        check_admin_referer('waolm_update_api_key');
        if (isset($_POST['waolm_server_api_key'])) {
			$options['waolm_server_api_key']=$_POST['waolm_server_api_key'];
            $x=is_multisite()?update_blog_option(1,'_waolm_server_options', $options):update_option('_waolm_server_options', $options);
        }
    }
}
//add_action('network_admin_menu', 'waolm_save_network_options');

