<?php
/**
 * Admin (Network): Gestión de Productos (WAOLM)
 * - Tabla: wp_waolm_product (SKU único, base_prefix)
 * - UI: listado, ordenar, paginación, buscar, añadir/editar/borrar
 */

if ( ! defined( 'ABSPATH' ) ) exit;

global $wpdb;

/** =========================
 *  Capacidad única (consistente)
 *  ========================= */
if ( ! defined('WAOLM_PRODUCTS_CAP') ) {
    define('WAOLM_PRODUCTS_CAP', 'edit_others_products'); // la que usas en el submenú
}

/** =========================
 *  Helper de URL admin (network-aware)
 *  ========================= */
function waolm_admin_base_url( $path = '' ) {
    // En esta pantalla estamos en Network Admin, pero lo dejamos genérico
    return is_network_admin()
        ? network_admin_url( ltrim($path, '/') )
        : admin_url( ltrim($path, '/') );
}

/** =========================
 *  DB: Crear tabla en base_prefix (nivel red)
 *  ========================= */
function waolm_products_install_table_network() {
    global $wpdb;
    $table   = ( is_multisite() ? $wpdb->base_prefix : $wpdb->prefix ) . 'waolm_product';
    $charset = $wpdb->get_charset_collate();

    require_once ABSPATH . 'wp-admin/includes/upgrade.php';
    $sql = "CREATE TABLE {$table} (
        id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
        product_name VARCHAR(255) NOT NULL,
        sku VARCHAR(100) NOT NULL,
        sort_order INT NOT NULL DEFAULT 0,
        active TINYINT(1) NOT NULL DEFAULT 1,
        created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        UNIQUE KEY sku (sku),
        PRIMARY KEY (id)
    ) {$charset};";
    dbDelta( $sql );
}
// Garantiza existencia al entrar al admin (network y site)
add_action('admin_init', function(){
    if ( current_user_can(WAOLM_PRODUCTS_CAP) ) {
        waolm_products_install_table_network();
    }
});

/** =========================
 *  Menú en Network Admin
 *  ========================= */
add_action('network_admin_menu', function () {
    add_submenu_page(
        'waolm_licensemanager_admin',   // tu menú principal de red
        'Productos WAOLM',              // Título de la página
        'Productos WAOLM',              // Título del menú
        WAOLM_PRODUCTS_CAP,             // Capacidad
        'waolm-products',               // Slug
        'waolm_products_render_page'    // Render
    );
});

/** =========================
 *  Screen options (per_page)
 *  ========================= */
add_filter('set-screen-option', function($status, $option, $value) {
    if ( 'waolm_products_per_page' === $option ) return (int) $value;
    return $status;
}, 10, 3);

function waolm_products_add_screen_option() {
    $screen = get_current_screen();
    if ( $screen && strpos($screen->id, 'waolm-products') !== false ) {
        add_screen_option('per_page', [
            'label'   => __('Productos por página', 'waolm'),
            'default' => 20,
            'option'  => 'waolm_products_per_page',
        ]);
    }
}
add_action('current_screen', 'waolm_products_add_screen_option');

/** =========================
 *  List table
 *  ========================= */
if ( ! class_exists('WP_List_Table') ) {
    require_once ABSPATH . 'wp-admin/includes/class-wp-list-table.php';
}

class WAOLM_Product_List_Table extends WP_List_Table {
    public function __construct() {
        parent::__construct([
            'singular' => 'waolm_product',
            'plural'   => 'waolm_products',
            'ajax'     => false,
        ]);
    }

    public function no_items() {
        esc_html_e('No hay productos.', 'waolm');
    }

    public function get_columns() {
        return [
            'product_name' => __('Nombre producto', 'waolm'),
            'sku'          => __('SKU', 'waolm'),
            'sort_order'   => __('Orden', 'waolm'),
            'active'       => __('Activo', 'waolm'),
            'actions'      => '', // botones a la derecha
        ];
    }

    public function get_sortable_columns() {
        return [
            'product_name' => ['product_name', true],
            'sku'          => ['sku', true],
            'sort_order'   => ['sort_order', true],
            'active'       => ['active', true],
        ];
    }

    public function column_default($item, $column_name) {
        switch ($column_name) {
            case 'product_name': return esc_html($item['product_name']);
            case 'sku':          return esc_html($item['sku']);
            case 'sort_order':   return (int) $item['sort_order'];
            case 'active':       return $item['active'] ? '✔' : '—';
            case 'actions':
                $edit_url = add_query_arg([
                    'page'   => 'waolm-products',
                    'action' => 'edit',
                    'id'     => (int)$item['id'],
                ], waolm_admin_base_url('admin.php'));

                $delete_url = wp_nonce_url(
                    add_query_arg([
						'page'   => 'waolm-products',
                        'action' => 'waolm_delete_product',
                        'id'     => (int)$item['id'],
                    ], waolm_admin_base_url('admin.php')),//admin-post 
                    'waolm_delete_product_'.$item['id']
                );

                $btn_edit = sprintf(
                    '<a class="button button-secondary" href="%s">%s</a>',
                    esc_url($edit_url), esc_html__('Editar', 'waolm')
                );
                $btn_delete = sprintf(
                    '<a class="button button-link-delete" href="%s" onclick="return confirm(\'%s\');">%s</a>',
                    esc_url($delete_url),
                    esc_attr(__('¿Está seguro de borrar este producto?', 'waolm')),
                    esc_html__('Borrar', 'waolm')
                );
                return '<div style="text-align:right; white-space:nowrap;">'.$btn_edit.' '.$btn_delete.'</div>';
        }
        return '';
    }

    public function prepare_items() {
        global $wpdb;
        $table_name = ( is_multisite() ? $wpdb->base_prefix : $wpdb->prefix ) . 'waolm_product';

        $per_page = (int) get_user_meta( get_current_user_id(), 'waolm_products_per_page', true );
        if ( $per_page <= 0 ) $per_page = 20;

        $orderby = isset($_GET['orderby']) ? sanitize_key($_GET['orderby']) : 'sort_order';
        $order   = (isset($_GET['order']) && 'desc' === strtolower($_GET['order'])) ? 'DESC' : 'ASC';
        $valid   = ['product_name','sku','sort_order','active','id'];
        if ( ! in_array($orderby, $valid, true) ) $orderby = 'sort_order';

        $paged    = max(1, (int)($_GET['paged'] ?? 1));
        $offset   = ($paged - 1) * $per_page;

        $search = isset($_GET['s']) ? wp_unslash(trim($_GET['s'])) : '';
        $where  = 'WHERE 1=1';
        $params = [];
        if ($search !== '') {
            $where .= ' AND (product_name LIKE %s OR sku LIKE %s)';
            $like   = '%' . $wpdb->esc_like($search) . '%';
            $params[] = $like;
            $params[] = $like;
        }

        $sql_total   = "SELECT COUNT(*) FROM {$table_name} {$where}";
        $total_items = (int) $wpdb->get_var( $params ? $wpdb->prepare($sql_total, $params) : $sql_total );

        $sql = "SELECT id, product_name, sku, sort_order, active
                FROM {$table_name}
                {$where}
                ORDER BY {$orderby} {$order}, id ASC
                LIMIT %d OFFSET %d";
        $query_params = $params;
        $query_params[] = $per_page;
        $query_params[] = $offset;

        $rows = $wpdb->get_results( $wpdb->prepare($sql, $query_params), ARRAY_A );
        $this->items = is_array($rows) ? $rows : [];

        $this->_column_headers = [$this->get_columns(), [], $this->get_sortable_columns(), 'product_name'];
        $this->set_pagination_args([
            'total_items' => $total_items,
            'per_page'    => $per_page,
            'total_pages' => max(1, (int) ceil($total_items / $per_page)),
        ]);
    }
}

/** =========================
 *  Handlers add/edit/delete (network-aware)
 *  ========================= */

// Guardar (crear/editar)
//add_action('admin_post_waolm_save_product', 
function save_product(){
    if ( ! current_user_can(WAOLM_PRODUCTS_CAP) ) {
        wp_die(__('No tienes permisos.', 'waolm'));
    }
    check_admin_referer('waolm_save_product');

    global $wpdb;
    $table = ( is_multisite() ? $wpdb->base_prefix : $wpdb->prefix ) . 'waolm_product';

    $id           = isset($_POST['id']) ? (int) $_POST['id'] : 0;
    $product_name = isset($_POST['product_name']) ? sanitize_text_field($_POST['product_name']) : '';
    $sku          = isset($_POST['sku']) ? sanitize_text_field($_POST['sku']) : '';
    $sort_order   = isset($_POST['sort_order']) ? (int) $_POST['sort_order'] : 0;
    $active       = isset($_POST['active']) ? 1 : 0;

    if ( $product_name === '' || $sku === '' ) {
        $redir = add_query_arg(['page'=>'waolm-products','action'=>'edit','id'=>$id,'msg'=>'empty'], waolm_admin_base_url('admin.php'));
        wp_safe_redirect( $redir ); exit;
    }

    $sku = trim($sku);

    if ( $id > 0 ) {
        $exists = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE sku=%s AND id<>%d", $sku, $id
        ));
        if ( $exists ) {
            $redir = add_query_arg(['page'=>'waolm-products','action'=>'edit','id'=>$id,'msg'=>'sku_exists'], waolm_admin_base_url('admin.php'));
            wp_safe_redirect( $redir ); exit;
        }

        $wpdb->update($table, [
            'product_name' => $product_name,
            'sku'          => $sku,
            'sort_order'   => $sort_order,
            'active'       => $active,
        ], ['id'=>$id], ['%s','%s','%d','%d'], ['%d']);
        $ok  = empty($wpdb->last_error);
        $msg = $ok ? 'updated' : 'dberror';
    } else {
        $exists = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$table} WHERE sku=%s", $sku
        ));
        if ( $exists ) {
            $redir = add_query_arg(['page'=>'waolm-products','action'=>'edit','msg'=>'sku_exists'], waolm_admin_base_url('admin.php'));
            wp_safe_redirect( $redir ); exit;
        }

        $wpdb->insert($table, [
            'product_name' => $product_name,
            'sku'          => $sku,
            'sort_order'   => $sort_order,
            'active'       => $active,
        ], ['%s','%s','%d','%d']);
        $ok  = empty($wpdb->last_error);
        $msg = $ok ? 'created' : 'dberror';
    }

    $redir = add_query_arg(['page'=>'waolm-products','msg'=>$msg], waolm_admin_base_url('admin.php'));
    wp_safe_redirect($redir); exit;
}

// Borrar
function delete_product(){
    if ( ! current_user_can(WAOLM_PRODUCTS_CAP) ) {
        wp_die(__('No tienes permisos.', 'waolm'));
    }
    $id = isset($_GET['id']) ? (int) $_GET['id'] : 0;
    check_admin_referer('waolm_delete_product_'.$id);

    if ( $id > 0 ) {
        global $wpdb;
        $table = ( is_multisite() ? $wpdb->base_prefix : $wpdb->prefix ) . 'waolm_product';
        $wpdb->delete($table, ['id'=>$id], ['%d']);
    }
    $redir = add_query_arg(['page'=>'waolm-products','msg'=>'deleted'], waolm_admin_base_url('admin.php'));
    wp_safe_redirect($redir); exit;
}

/** =========================
 *  Render de la pantalla
 *  ========================= */
function waolm_products_render_page() {
    if ( ! current_user_can(WAOLM_PRODUCTS_CAP) ) {
        wp_die(__('No tienes permisos.', 'waolm'));
    }

    $list_table = new WAOLM_Product_List_Table();
    $list_table->prepare_items();

    $is_edit = (isset($_GET['action']) && $_GET['action']==='edit');
    $editing = null;
	if(isset($_POST['action'])&&$_POST['action']=="waolm_save_product" ){
		save_product();
	}
	if(isset($_GET['action'])&&$_GET['action']=="waolm_delete_product" ){
		delete_product();
	}
    if ( $is_edit ) {
        global $wpdb;
        $table = ( is_multisite() ? $wpdb->base_prefix : $wpdb->prefix ) . 'waolm_product'; 
        $id    = isset($_REQUEST['id']) ? (int) $_REQUEST['id'] : 0;
        if ( $id > 0 ) {
			$q=$wpdb->prepare("SELECT * FROM {$table} WHERE id=%d", $id);
            $editing = $wpdb->get_row( $q, ARRAY_A );
        } else {
            $editing = ['id'=>0,'product_name'=>'','sku'=>'','sort_order'=>0,'active'=>1];
        }
    }
    $msg = isset($_GET['msg']) ? sanitize_key($_GET['msg']) : '';
    $notices = [
        'created'    => __('Producto creado correctamente.:)', 'waolm'),
        'updated'    => __('Producto actualizado correctamente.', 'waolm'),
        'deleted'    => __('Producto borrado.:(', 'waolm'),
        'dberror'    => __('Error en base de datos.', 'waolm'),
        'sku_exists' => __('El SKU ya existe. Debe ser único.', 'waolm'),
        'empty'      => __('Nombre y SKU son obligatorios.', 'waolm'),
    ];
    ?>
    <div class="wrap">
        <h1 class="wp-heading-inline"><?php esc_html_e('WAOLM Productos.', 'waolm'); ?></h1>

        <?php if ( $msg && isset($notices[$msg]) ): ?>
            <div class="notice notice-<?php echo $msg==='dberror' ? 'error' : 'success'; ?> is-dismissible">
                <p><?php echo esc_html($notices[$msg]); ?></p>
            </div>
        <?php endif; ?>

        <?php if ( ! $is_edit ): ?>
            <a href="<?php echo esc_url( add_query_arg(['page'=>'waolm-products','action'=>'edit'], waolm_admin_base_url('admin.php')) ); ?>" class="page-title-action">
                <?php esc_html_e('Añadir nuevo producto', 'waolm'); ?>
            </a>
        <?php endif; ?>

        <?php if ( $is_edit && is_array($editing) ): ?>
            <hr class="wp-header-end">
            <h2><?php echo ( (int)$editing['id'] === 0 ) ? esc_html__('Añadir producto', 'waolm') : esc_html__('Editar producto', 'waolm'); ?></h2>
			<?php $home = is_multisite()?get_blog_option(1,"home"):get_option("home"); //aqui ?>
            <form method="post" action="<?php echo $home; ?>/wp-admin/network/admin.php?page=waolm-products">
                <?php wp_nonce_field('waolm_save_product'); ?>
                <input type="hidden" name="action" value="waolm_save_product">
                <input type="hidden" name="id"     value="<?php echo (int)$editing['id']; ?>">

                <table class="form-table" role="presentation">
                    <tbody>
                        <tr>
                            <th scope="row"><label for="waolm_product_name"><?php esc_html_e('Nombre producto', 'waolm'); ?></label></th>
                            <td><input name="product_name" type="text" id="waolm_product_name" value="<?php echo esc_attr($editing['product_name']); ?>" class="regular-text" required></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="waolm_sku"><?php esc_html_e('SKU', 'waolm'); ?></label></th>
                            <td><input name="sku" type="text" id="waolm_sku" value="<?php echo esc_attr($editing['sku']); ?>" class="regular-text" required></td>
                        </tr>
                        <tr>
                            <th scope="row"><label for="waolm_sort_order"><?php esc_html_e('Orden', 'waolm'); ?></label></th>
                            <td><input name="sort_order" type="number" id="waolm_sort_order" value="<?php echo (int)$editing['sort_order']; ?>" class="small-text"></td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Activo', 'waolm'); ?></th>
                            <td><label><input name="active" type="checkbox" value="1" <?php checked( (int)$editing['active'], 1 ); ?>> <?php esc_html_e('Sí', 'waolm'); ?></label></td>
                        </tr>
                    </tbody>
                </table>

                <?php submit_button( ( (int)$editing['id'] === 0 ) ? __('Guardar', 'waolm') : __('Actualizar', 'waolm') ); ?>
                <a href="<?php echo esc_url( add_query_arg(['page'=>'waolm-products'], waolm_admin_base_url('admin.php')) ); ?>" class="button button-secondary">
                    <?php esc_html_e('Cancelar', 'waolm'); ?>
                </a>
            </form>
            <hr>
        <?php endif; ?>

        <?php if ( ! $is_edit ): ?>
            <form method="get">
                <input type="hidden" name="page" value="waolm-products">
                <?php $list_table->search_box( __('Buscar SKU o nombre', 'waolm'), 'waolm-products' ); ?>
                <?php $list_table->display(); ?>
            </form>
        <?php endif; ?>
    </div>
    <?php
}



function poblar_productos() {
    // Bandera "una vez"
    if ( is_multisite() ) {
        if ( get_site_option('waolm_products_seeded') ) return;
    } else {
        if ( get_option('waolm_products_seeded') ) return;
    }

    global $wpdb;
    $table = ( is_multisite() ? $wpdb->base_prefix : $wpdb->prefix ) . 'waolm_product';

    // Asegura la tabla
    waolm_products_install_table_network();

    // Lista: [nombre, sku] en el orden exacto
    $data = [
        ['Microsoft Office 2024 Pro','O24PP'],
        ['Microsoft Office 2021 Pro','O21PP'],
        ['Microsoft Office 2019 Pro','O19PP'],
        ['Microsoft Office 2016 Pro','O16PP'],
        ['Microsoft Office 2013 Pro','O13PP'],
        ['Microsoft Office 2010 Pro','O10PP'],
        ['Microsoft Office 2024 Standard','O24S'],
        ['Microsoft Office 2021 Standard','O21S'],
        ['Microsoft Office 2019 Standard','O19S'],
        ['Microsoft Office 2016 Standard','O16S'],
        ['Microsoft Office 2013 Standard','O13S'],
        ['Microsoft Office 2010 Standard','O10S'],
        ['Microsoft Visio 2024 Pro','V24P'],
        ['Microsoft Visio 2019 Pro','V19P'],
        ['Microsoft Visio 2016 Pro','V16P'],
        ['Microsoft Visio 2013 Pro','V13P'],
        ['Microsoft Visio 2010 Pro','V10P'],
        ['Microsoft Visio 2024 Standard','V24S'],
        ['Microsoft Visio 2021 Standard','V21S'],
        ['Microsoft Visio 2019 Standard','V19S'],
        ['Microsoft Visio 2016 Standard','V16S'],
        ['Microsoft Visio 2013 Standard','V13S'],
        ['Microsoft Visio 2010 Standard','V10S'],
        ['Microsoft Project 2024 Pro','P24P'],
        ['Microsoft Project 2021 Pro','P21P'],
        ['Microsoft Project 2019 Pro','P19P'],
        ['Microsoft Project 2016 Pro','P16P'],
        ['Microsoft Project 2013 Pro','P13P'],
        ['Microsoft Project 2010 Pro','P10P'],
        ['Microsoft Project 2024 Standard','P24S'],
        ['Microsoft Project 2019 Standard','P19S'],
        ['Microsoft Project 2016 Standard','P16S'],
        ['Microsoft Project 2013 Standard','P13S'],
        ['Microsoft Project 2010 Standard','P10S'],
        ['Windows 11 Pro','W11P'],
        ['Windows 11 Home','W11H'],
        ['Windows 10 Pro','W10P'],
        ['Windows 10 Home','W10H'],
    ];

    // Inserta/actualiza con sort_order incremental y activo=1
    $sort = 1;
    foreach ( $data as [$name, $sku] ) {
        // ON DUPLICATE evita error por UNIQUE(sku) y actualiza campos clave
        $wpdb->query(
            $wpdb->prepare(
                "INSERT INTO {$table} (product_name, sku, sort_order, active)
                 VALUES (%s, %s, %d, 1)
                 ON DUPLICATE KEY UPDATE
                    product_name = VALUES(product_name),
                    sort_order   = VALUES(sort_order),
                    active       = VALUES(active)",
                $name, $sku, $sort
            )
        );
        $sort++;
    }

    // Marca como ejecutado una vez
    if ( is_multisite() ) {
        add_site_option('waolm_products_seeded', time());
    } else {
        add_option('waolm_products_seeded', time());
    }
}

add_action('admin_init', 'poblar_productos');
