<?php
/**
 * softa functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package softa
 */

if ( ! function_exists( 'softa_setup' ) ) :
/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function softa_setup() {
	/*
	 * Make theme available for translation.
	 * Translations can be filed in the /languages/ directory.
	 * If you're building a theme based on softa, use a find and replace
	 * to change 'softa' to the name of your theme in all the template files.
	 */
	load_theme_textdomain( 'softa', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/*
	 * Enable support for Post Thumbnails on posts and pages.
	 *
	 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
	 */
	add_theme_support( 'post-thumbnails' );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
		'navigation' => esc_html__( 'Primary navigation', 'softa' ),
	) );

	// Add theme support for selective refresh for widgets.
	add_theme_support( 'customize-selective-refresh-widgets' );

	// Add theme support for WooCommerce.
	add_theme_support( 'woocommerce' );

	if ( ! isset( $content_width ) ) $content_width = 900;
}
endif;
add_action( 'after_setup_theme', 'softa_setup' ); 

/*
Remove default woocommerce css
*/
add_filter( 'woocommerce_enqueue_styles', '__return_false' );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function softa_widgets_init() {
	register_sidebar( array(
		'name'          => esc_html__( 'Shop Sidebar', 'softa' ),
		'id'            => 'shop_sidebar',
		'description'   => esc_html__( 'Add widgets here.', 'softa' ),
		'before_widget' => '<div class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<div class="heading-decor-s tal style2"><h6 class="h">',
		'after_title'   => '</h6></div>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Product Single Sidebar', 'softa' ),
		'id'            => 'product_single_sidebar',
		'description'   => esc_html__( 'Add widgets here.', 'softa' ),
		'before_widget' => '<div class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<div class="heading-decor-s tal style2"><h6 class="h">',
		'after_title'   => '</h6></div>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer col 1', 'softa' ),
		'id'            => 'footer-1',
		'description'   => esc_html__( 'Add widgets here.', 'softa' ),
		'before_widget' => '<div class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h6 class="widget-title">',
		'after_title'   => '</h6>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer col 2', 'softa' ),
		'id'            => 'footer-2',
		'description'   => esc_html__( 'Add widgets here.', 'softa' ),
		'before_widget' => '<div class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h6 class="widget-title">',
		'after_title'   => '</h6>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer col 3', 'softa' ),
		'id'            => 'footer-3',
		'description'   => esc_html__( 'Add widgets here.', 'softa' ),
		'before_widget' => '<div class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h6 class="widget-title">',
		'after_title'   => '</h6>',
	) );
	register_sidebar( array(
		'name'          => esc_html__( 'Footer col 4', 'softa' ),
		'id'            => 'footer-4',
		'description'   => esc_html__( 'Add widgets here.', 'softa' ),
		'before_widget' => '<div class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h6 class="widget-title">',
		'after_title'   => '</h6>',
	) );
}
add_action( 'widgets_init', 'softa_widgets_init' );

/**
 * Add Google fonts.
 */
if(!class_exists('WPBakeryShortCode')) {
	function softa_fonts_url() {
	    $font_url = '';
	    if ( 'off' !== _x( 'on', 'Google font: on or off', 'softa' ) ) {
	        $font_url = add_query_arg( 'family', 'Montserrat:100,100i,200,200i,300,300i,400,400i,500,500i,600,600i,700,700i,800,800i,900,900i', "//fonts.googleapis.com/css" );
	    }
	    return $font_url;
	}
}

/**
 * Enqueue scripts and styles.
 */
function softa_scripts() {
	wp_enqueue_style( 'softa-style-default', get_stylesheet_uri() );

	if(!class_exists('WPBakeryShortCode')) {
		wp_enqueue_style( 'softa-fonts', softa_fonts_url(), array(), '1.0.0' );
	}

	/* if(function_exists('vc_icon_element_fonts_enqueue')) {
		vc_icon_element_fonts_enqueue( 'fontawesome' );
	} else { */
		wp_enqueue_style( 'softa-font-awesomesd', get_template_directory_uri() . '/css/font-awesome.min.css');
		wp_enqueue_style( 'softa-iconsss', get_template_directory_uri() . '/css/iconfont.css');
	//}

	wp_enqueue_style( 'softa-frontend-grid', get_template_directory_uri() . '/css/frontend-grid.css');
	wp_style_add_data('softa-frontend-grid', 'rtl', 'replace');

	wp_enqueue_style( 'woocommerce-general', get_template_directory_uri() . '/css/woocommerce.css');
	wp_style_add_data('woocommerce-general', 'rtl', 'replace');

	wp_enqueue_style( 'woocommerce-layout', get_template_directory_uri() . '/css/woocommerce-layout.css');
	wp_style_add_data('woocommerce-layout', 'rtl', 'replace');

	wp_enqueue_style( 'photoswipe', get_template_directory_uri() . '/css/photoswipe.css');

	wp_enqueue_style( 'photoswipe-default-skin', get_template_directory_uri() . '/css/default-skin.css');

	wp_enqueue_style( 'softa-circle-animations', get_template_directory_uri() . '/css/circle_animations.css');

	wp_enqueue_style( 'softa-style', get_template_directory_uri() . '/css/style.css');
	wp_style_add_data('softa-style', 'rtl', 'replace');

	wp_add_inline_style( 'softa-style', do_action('inline_css') );

	wp_enqueue_style( 'softa-mobile', get_template_directory_uri() . '/css/mobile.css');
	wp_style_add_data('softa-mobile', 'rtl', 'replace');

	wp_enqueue_script( 'softa-skip-link-focus-fix', get_template_directory_uri() . '/js/skip-link-focus-fix.js', array('jquery'), '', true );

	wp_enqueue_script( 'softa-load-posts', get_template_directory_uri() . '/js/load-posts.js', array('jquery'), '', true );

	wp_enqueue_script( 'imagesloaded' );

	wp_enqueue_script( 'isotope', get_template_directory_uri() . '/js/isotope.pkgd.min.js', array('jquery'), '', true );

	wp_enqueue_script( 'photoswipe', get_template_directory_uri() . '/js/photoswipe.min.js', array('jquery'), '', true );

	wp_enqueue_script( 'photoswipe-ui-default', get_template_directory_uri() . '/js/photoswipe-ui-default.min.js', array('jquery'), '', true );

	wp_enqueue_script( 'softa-script', get_template_directory_uri() . '/js/script.js', array('jquery'), '', true );

	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}
add_action( 'wp_enqueue_scripts', 'softa_scripts', 100 );

function softa_add_editor_styles() {
	add_editor_style( get_template_directory_uri() . '/css/style.css' );
}
add_action( 'current_screen', 'softa_add_editor_styles' );

add_action( 'admin_enqueue_scripts', 'load_admin_styles', 1000 );
function load_admin_styles() {
	wp_enqueue_style( 'softa-font-awesomesd', get_template_directory_uri() . '/css/font-awesome.min.css');
	wp_enqueue_style( 'softa-backend', get_template_directory_uri() . '/css/admin.css', array(), '1.0', false );
	wp_enqueue_style( 'softa-shortcode-icons', get_template_directory_uri() . '/css/shortcode-icons.css', array(), '1.0', false );

  wp_enqueue_script('softa-admin', get_parent_theme_file_uri() . '/js/admin.js', array('jquery'), null, true);
}

/**
 * Admin Pages
 */
require get_template_directory() . '/inc/admin-pages.php';

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Custom functions that act independently of the theme templates.
 */
require get_template_directory() . '/inc/extras.php';

/**
 * Load Jetpack compatibility file.
 */
require get_template_directory() . '/inc/jetpack.php';

/**
 * Load TGM file.
 */
require get_template_directory() . '/tgm/tgm.php';

/**
 * Load ACF.
 */
require get_template_directory() . '/inc/acf.php';
define( 'ACF_LITE', true );

/**
 * Load Theme Settings.
 */
require get_template_directory() . '/theme-settings/config.php';

/**
 * Hooks
 */
require get_template_directory() . '/inc/v-hook.php';

/**
 * Setup Wizard
 */

if (is_admin()) {
  require_once get_template_directory() . '/inc/setup-wizard/envato_setup_init.php';
  require_once get_template_directory() . '/inc/setup-wizard/envato_setup.php';
}

/**
 * Site pagination.
 */
function softa_wp_corenavi($max_count = '') {
	global $wp_query;
	$pages = '';
	if(isset($max_count) && $max_count > 0) {
		$max = $max_count;
	} else {
		$max = $wp_query->max_num_pages;
	}

	if(get_query_var('paged') != 0) {
		$paged = get_query_var('paged');
	} else {
		$paged = get_query_var('page');
	}

	if (!$current = $paged) $current = 1;
	$a['base'] = str_replace(999999999, '%#%', get_pagenum_link(999999999));
	$a['total'] = $max;
	$a['current'] = $current;

	$a['mid_size'] = 5;
	$a['end_size'] = 1;
	$a['prev_text'] = '<i class="basic-ui-icon-left-arrow"></i>';
	$a['next_text'] = '<i class="basic-ui-icon-right-arrow"></i>';

	$html = "";
	if ($max > 1) $html .= '<div class="pagination col-xs-12">';
	$html .= paginate_links($a);
	if ($max > 1) $html .= '</div>';

	return $html;
}

add_action( 'admin_init', 'softa_hide_editor' );

function softa_hide_editor() {

        // Get the Post ID.
        if ( isset ( $_GET['post'] ) )
        $post_id = $_GET['post'];
        else if ( isset ( $_POST['post_ID'] ) )
        $post_id = $_POST['post_ID'];

    if( !isset ( $post_id ) || empty ( $post_id ) )
        return;

    // Get the name of the Page Template file.
    $template_file = get_post_meta($post_id, '_wp_page_template', true);

    if($template_file == 'page-coming-soon.php'){ // edit the template name
        remove_post_type_support('page', 'editor');
    }

}

add_filter('get_the_excerpt', 'shortcode_unautop');
add_filter('get_the_excerpt', 'do_shortcode');

add_action( 'vc_before_init', 'softa_vc_before_init_actions' );
 
function softa_vc_before_init_actions() {
    if( function_exists('vc_set_shortcodes_templates_dir') ){ 
        vc_set_shortcodes_templates_dir( get_template_directory() . '/vc-elements' );
    }
}

function softa_styles() {
	global $softa_theme;

	$style = array();
	$style['right_click_disable'] = '0';
	$style['right_click_disable_message'] = __('<p style="text-align: center"><strong><span style="font-size: 18px">Content is protected. Right-click function is disabled.</span></strong></p>', 'softa');
	$style['protected_message'] = esc_html__( 'This content is password protected. To view it please enter your password below:', 'softa' );

	$style['header_style'] = 'logo_left';
	$style['header_color_mode'] = 'light';
	$style['header_container'] = 'container-fluid';
	$style['navigation_type'] = 'visible_menu';
	$style['css_classes'] = '';

	$style['header_button_show'] = '1';
	$style['button_text'] = '';
	$style['button_link'] = '';

	$style['logo_content'] = get_bloginfo( 'name' );
	$style['logo_variant'] = 'default';
	$style['logo_default_url'] = '';
	$style['logo_light_url'] = '';
	$style['logo_dark_url']  = '';

	$style['logo2_light_url'] = '';
	$style['logo2_dark_url']  = '';
	$style['logo2_content']  = '';

	$style['logo3_light_url'] = '';
	$style['logo3_dark_url']  = '';
	$style['logo3_content']  = '';

	$style['logo4_light_url'] = '';
	$style['logo4_dark_url']  = '';
	$style['logo4_content']  = '';

	$style['color_mode'] = 'dark';
	$style['copyright_text']  = '';

	$style['search'] = 'yes';
	$style['cart'] = 'yes';
	$style['header_space'] = 'yes';
	$style['footer'] = 'show';

	$style['social_target'] = '_self';
	$style['facebook_link'] = '';
	$style['twitter_link'] = '';
	$style['instagram_link'] = '';
	$style['youtube_link'] = '';
	$style['gp_link'] = '';
	$style['linkedin_link'] = '';
	$style['skype_link'] = '';
	$style['telegram_link'] = '';

	$style['google_maps_api_key'] = '';

	$style['footer_col_1'] = '3';
	$style['footer_col_2'] = '3';
	$style['footer_col_3'] = '3';
	$style['footer_col_4'] = '3';

	$style['show_footer_top'] = '1';
	$style['show_subcribe_form'] = '1';
	$style['subcribe_code'] = '';
	$style['show_social_buttons'] = '1';
	$style['show_search_form'] = '1';

	$style['404_bg'] = '';
	$style['404_heading'] = esc_html__('404 error', 'softa');
	$style['404_text'] = esc_html__('You are on a non-existing page', 'softa');

	$style['coming_soon_bg'] = '';
	$style['coming_soon_subscribe_code'] = '';
	$style['coming_soon_heading'] = esc_html__('Coming Soon', 'softa');
	$style['coming_soon_text'] = esc_html__('Subscribe and get the latest updates', 'softa');

	$style['project_style'] = 'slider';
	$style['project_image'] = 'full';
	$style['project_date'] = 'show';
	$style['project_categories'] = 'show';

	$style['blog_feature_image'] = 'show';
	$style['blog_date'] = 'show';
	$style['blog_categories'] = 'show';

	$style['preloader_show'] = '0';
	$style['preloader_img'] = '';

	$style['code_in_head'] = '';
	$style['code_before_body'] = '';

	if ( isset($softa_theme) && !empty($softa_theme) ) {
		if(isset($softa_theme['show_search']) && $softa_theme['show_search'] == '1') {
			$style['search'] = 'yes';
		} else {
			$style['search'] = 'no';
		}

		if(isset($softa_theme['show_cart']) && $softa_theme['show_cart'] == '1') {
			$style['cart'] = 'yes';
		} else {
			$style['cart'] = 'no';
		}

		if(isset($softa_theme['default_logo']['background-image']) && $softa_theme['default_logo']['background-image']) { 
			$style['logo_default_url'] = $softa_theme['default_logo']['background-image'];
		} 
		if(isset($softa_theme['dark_logo']['background-image']) && $softa_theme['dark_logo']['background-image']) {
			$style['logo_dark_url'] = $softa_theme['dark_logo']['background-image'];
		} 
 		if(isset($softa_theme['light_logo']['background-image']) && $softa_theme['light_logo']['background-image']) {
			$style['logo_light_url'] = $softa_theme['light_logo']['background-image'];
		} 

		if(!empty($style['logo_light_url']) && !empty($style['logo_dark_url'])) {
			$style['logo_content'] = '<img class="light" src="'.esc_url($style['logo_light_url']).'" alt="'.get_bloginfo( 'name' ).'"><img class="dark" src="'.esc_url($style['logo_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
		} elseif(!empty($style['logo_light_url']) || !empty($style['logo_dark_url'])) {
			if(!empty($style['logo_light_url'])) {
				$style['logo_content'] = '<img src="'.esc_url($style['logo_light_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
			if(!empty($style['logo_dark_url'])) {
				$style['logo_content'] = '<img src="'.esc_url($style['logo_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
		} elseif(!empty($style['logo_default_url'])) {
			$style['logo_content'] = '<img src="'.esc_url($style['logo_default_url']).'" alt="'.get_bloginfo( 'name' ).'">';
		} elseif(isset($softa_theme['logo_text']) && $softa_theme['logo_text']) {
			$style['logo_content'] = esc_html($softa_theme['logo_text']);
		} else {
			$style['logo_content'] = get_bloginfo( 'name' );
		}

		if(isset($softa_theme['dark2_logo']['background-image']) && $softa_theme['dark2_logo']['background-image']) {
			$style['logo2_dark_url'] = $softa_theme['dark2_logo']['background-image'];
		} 
 		if(isset($softa_theme['light2_logo']['background-image']) && $softa_theme['light2_logo']['background-image']) {
			$style['logo2_light_url'] = $softa_theme['light2_logo']['background-image'];
		} 

		if(!empty($style['logo2_light_url']) && !empty($style['logo2_dark_url'])) {
			$style['logo2_content'] = '<img class="light" src="'.esc_url($style['logo2_light_url']).'" alt="'.get_bloginfo( 'name' ).'"><img class="dark" src="'.esc_url($style['logo2_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
		} elseif(!empty($style['logo2_light_url']) || !empty($style['logo2_dark_url'])) {
			if(!empty($style['logo2_light_url'])) {
				$style['logo2_content'] = '<img src="'.esc_url($style['logo2_light_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
			if(!empty($style['logo2_dark_url'])) {
				$style['logo2_content'] = '<img src="'.esc_url($style['logo2_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
		}

		if(isset($softa_theme['dark3_logo']['background-image']) && $softa_theme['dark3_logo']['background-image']) {
			$style['logo3_dark_url'] = $softa_theme['dark3_logo']['background-image'];
		} 
 		if(isset($softa_theme['light3_logo']['background-image']) && $softa_theme['light3_logo']['background-image']) {
			$style['logo3_light_url'] = $softa_theme['light3_logo']['background-image'];
		} 

		if(!empty($style['logo3_light_url']) && !empty($style['logo3_dark_url'])) {
			$style['logo3_content'] = '<img class="light" src="'.esc_url($style['logo3_light_url']).'" alt="'.get_bloginfo( 'name' ).'"><img class="dark" src="'.esc_url($style['logo3_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
		} elseif(!empty($style['logo3_light_url']) || !empty($style['logo3_dark_url'])) {
			if(!empty($style['logo3_light_url'])) {
				$style['logo3_content'] = '<img src="'.esc_url($style['logo3_light_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
			if(!empty($style['logo2_dark_url'])) {
				$style['logo3_content'] = '<img src="'.esc_url($style['logo3_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
		}

		if(isset($softa_theme['dark4_logo']['background-image']) && $softa_theme['dark4_logo']['background-image']) {
			$style['logo4_dark_url'] = $softa_theme['dark4_logo']['background-image'];
		} 
 		if(isset($softa_theme['light4_logo']['background-image']) && $softa_theme['light4_logo']['background-image']) {
			$style['logo4_light_url'] = $softa_theme['light4_logo']['background-image'];
		} 

		if(!empty($style['logo4_light_url']) && !empty($style['logo4_dark_url'])) {
			$style['logo4_content'] = '<img class="light" src="'.esc_url($style['logo4_light_url']).'" alt="'.get_bloginfo( 'name' ).'"><img class="dark" src="'.esc_url($style['logo4_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
		} elseif(!empty($style['logo4_light_url']) || !empty($style['logo4_dark_url'])) {
			if(!empty($style['logo4_light_url'])) {
				$style['logo4_content'] = '<img src="'.esc_url($style['logo4_light_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
			if(!empty($style['logo4_dark_url'])) {
				$style['logo4_content'] = '<img src="'.esc_url($style['logo4_dark_url']).'" alt="'.get_bloginfo( 'name' ).'">';
			}
		}

		if(isset($softa_theme['header_style']) && !empty($softa_theme['header_style'])) {
			$style['header_style'] = $softa_theme['header_style'];
		}

		if(isset($softa_theme['header_container']) && !empty($softa_theme['header_container'])) {
			$style['header_container'] = $softa_theme['header_container'];
		}

		if(isset($softa_theme['google_maps_api_key']) && !empty($softa_theme['google_maps_api_key'])) {
			$style['google_maps_api_key'] = $softa_theme['google_maps_api_key'];
		}

		if(isset($softa_theme['header_button_show'])) {
			$style['header_button_show'] = $softa_theme['header_button_show'];
		}
		
		if(isset($softa_theme['button_text']) && !empty($softa_theme['button_text'])) {
			$style['button_text'] = $softa_theme['button_text'];
		}

		if(isset($softa_theme['button_link']) && !empty($softa_theme['button_link'])) {
			$style['button_link'] = $softa_theme['button_link'];
		}

		if(isset($softa_theme['header_color_mode']) && !empty($softa_theme['header_color_mode'])) {
			$style['header_color_mode'] = $softa_theme['header_color_mode'];
		}

		if(isset($softa_theme['right_click_disable'])) {
			$style['right_click_disable'] = $softa_theme['right_click_disable'];
		}

		if(isset($softa_theme['right_click_disable_message']) && !empty($softa_theme['right_click_disable_message'])) {
			$style['right_click_disable_message'] = $softa_theme['right_click_disable_message'];
		}

		if(isset($softa_theme['protected_message']) && !empty($softa_theme['protected_message'])) {
			$style['protected_message'] = $softa_theme['protected_message'];
		}

		if(isset($softa_theme['copyright_text']) && !empty($softa_theme['copyright_text'])) {
			$style['copyright_text'] = $softa_theme['copyright_text'];
		}

		if(isset($softa_theme['social_target']) && $softa_theme['social_target']) {
			$style['social_target'] = $softa_theme['social_target'];
		}

		if(isset($softa_theme['facebook_link']) && $softa_theme['facebook_link']) {
			$style['facebook_link'] = $softa_theme['facebook_link'];
		}
		if(isset($softa_theme['twitter_link']) && $softa_theme['twitter_link']) {
			$style['twitter_link'] = $softa_theme['twitter_link'];
		}
		if(isset($softa_theme['instagram_link']) && $softa_theme['instagram_link']) {
			$style['instagram_link'] = $softa_theme['instagram_link'];
		}
		if(isset($softa_theme['youtube_link']) && $softa_theme['youtube_link']) {
			$style['youtube_link'] = $softa_theme['youtube_link'];
		}
		if(isset($softa_theme['gp_link']) && $softa_theme['gp_link']) {
			$style['gp_link'] = $softa_theme['gp_link'];
		}
		if(isset($softa_theme['linkedin_link']) && $softa_theme['linkedin_link']) {
			$style['linkedin_link'] = $softa_theme['linkedin_link'];
		}
		if(isset($softa_theme['skype_link']) && $softa_theme['skype_link']) {
			$style['skype_link'] = $softa_theme['skype_link'];
		}
		if(isset($softa_theme['telegram_link']) && $softa_theme['telegram_link']) {
			$style['telegram_link'] = $softa_theme['telegram_link'];
		}

		if(isset($softa_theme['footer']) && $softa_theme['footer']) {
			$style['footer'] = $softa_theme['footer'];
		}

		if(isset($softa_theme['footer_col_1']) && $softa_theme['footer_col_1']) {
			$style['footer_col_1'] = $softa_theme['footer_col_1'];
		}

		if(isset($softa_theme['footer_col_2']) && $softa_theme['footer_col_2']) {
			$style['footer_col_2'] = $softa_theme['footer_col_2'];
		}

		if(isset($softa_theme['footer_col_3']) && $softa_theme['footer_col_3']) {
			$style['footer_col_3'] = $softa_theme['footer_col_3'];
		}

		if(isset($softa_theme['footer_col_4']) && $softa_theme['footer_col_4']) {
			$style['footer_col_4'] = $softa_theme['footer_col_4'];
		}

		if(isset($softa_theme['404_bg']) && !empty($softa_theme['404_bg'])) {
			$style['404_bg'] = "background-image: url(".$softa_theme['404_bg']['background-image'].")";
		}

		if(isset($softa_theme['404_heading']) && $softa_theme['404_heading']) {
			$style['404_heading'] = $softa_theme['404_heading'];
		}

		if(isset($softa_theme['404_text']) && $softa_theme['404_text']) {
			$style['404_text'] = $softa_theme['404_text'];
		}

		if(isset($softa_theme['coming_soon_bg']) && !empty($softa_theme['coming_soon_bg'])) {
			$style['coming_soon_bg'] = "background-image: url(".$softa_theme['coming_soon_bg']['background-image'].")";
		}

		if(isset($softa_theme['coming_soon_heading']) && $softa_theme['coming_soon_heading']) {
			$style['coming_soon_heading'] = $softa_theme['coming_soon_heading'];
		}

		if(isset($softa_theme['coming_soon_text']) && $softa_theme['coming_soon_text']) {
			$style['coming_soon_text'] = $softa_theme['coming_soon_text'];
		}

		if(isset($softa_theme['coming_soon_subscribe_code']) && $softa_theme['coming_soon_subscribe_code']) {
			$style['coming_soon_subscribe_code'] = $softa_theme['coming_soon_subscribe_code'];
		}

		if(isset($softa_theme['project_style']) && $softa_theme['project_style']) {
			$style['project_style'] = $softa_theme['project_style'];
		}

		if(isset($softa_theme['project_image']) && $softa_theme['project_image']) {
			$style['project_image'] = $softa_theme['project_image'];
		}
		
		if(isset($softa_theme['project_date']) && $softa_theme['project_date']) {
			$style['project_date'] = $softa_theme['project_date'];
		}
		
		if(isset($softa_theme['project_categories']) && $softa_theme['project_categories']) {
			$style['project_categories'] = $softa_theme['project_categories'];
		}

		if(isset($softa_theme['code_in_head']) && $softa_theme['code_in_head']) {
			$style['code_in_head'] = $softa_theme['code_in_head'];
		}
		
		if(isset($softa_theme['code_before_body']) && $softa_theme['code_before_body']) {
			$style['code_before_body'] = $softa_theme['code_before_body'];
		}

		if(isset($softa_theme['preloader_show']) && $softa_theme['preloader_show']) {
			$style['preloader_show'] = $softa_theme['preloader_show'];
		}
		
		if(isset($softa_theme['preloader_type']) && $softa_theme['preloader_type']) {
			$style['preloader_type'] = $softa_theme['preloader_type'];
		}
		
		if(isset($softa_theme['preloader_img']) && $softa_theme['preloader_img']) {
			$style['preloader_img'] = $softa_theme['preloader_img'];
		}
		
		if(isset($softa_theme['show_footer_top']) && $softa_theme['show_footer_top'] == 0) {
			$style['show_footer_top'] = $softa_theme['show_footer_top'];
		}

		if(isset($softa_theme['show_subcribe_form']) && $softa_theme['show_subcribe_form'] == 0) {
			$style['show_subcribe_form'] = $softa_theme['show_subcribe_form'];

		}
		
		if(isset($softa_theme['subcribe_code']) && $softa_theme['subcribe_code']) {
			$style['subcribe_code'] = $softa_theme['subcribe_code'];
		}

		if(isset($softa_theme['show_social_buttons']) && $softa_theme['show_social_buttons'] == 0) {
			$style['show_social_buttons'] = $softa_theme['show_social_buttons'];
		}
		
		if(isset($softa_theme['show_search_form']) && $softa_theme['show_search_form'] == 0) {
			$style['show_search_form'] = $softa_theme['show_search_form'];
		}
		
		if(isset($softa_theme['blog_feature_image']) && $softa_theme['blog_feature_image']) {
			$style['blog_feature_image'] = $softa_theme['blog_feature_image'];
		}
		
		if(isset($softa_theme['blog_date']) && $softa_theme['blog_date']) {
			$style['blog_date'] = $softa_theme['blog_date'];
		}
		
		if(isset($softa_theme['blog_categories']) && $softa_theme['blog_categories']) {
			$style['blog_categories'] = $softa_theme['blog_categories'];
		}

	}

	if(function_exists("get_field")) {

		if(get_field('header_container') == 'container') {
			$style['header_container'] = 'container';
		} elseif(get_field('header_container') == 'container-fluid') {
			$style['header_container'] = 'container-fluid';
		}

		if(get_field('header_color_mode') == 'light') {
			$style['header_color_mode'] = 'light';
		}

		if(get_field('header_color_mode') == 'dark') {
			$style['header_color_mode'] = 'dark';
		}

		if(get_field('header_space')) {
			$style['header_space'] = get_field('header_space');
		}

		if(get_field('header_style') && get_field('header_style') != 'default') {
			$style['header_style'] = get_field('header_style');
		}

		if(get_field('navigation_type') && get_field('navigation_type') != 'default') {
			$style['navigation_type'] = get_field('navigation_type');
		}

		if(get_field('search_m') && get_field('search_m') != 'default') {
			$style['search'] = get_field('search_m');
		}

		if(get_field('cart') && get_field('cart') != 'default') {
			$style['cart'] = get_field('cart');
		}

		if(get_field('header_container') && get_field('header_container') != 'default') {
			$style['header_container'] = get_field('header_container');
		}

		if(get_field('footer_social_buttons') && get_field('footer_social_buttons') != 'default') {
			$style['footer_social_buttons'] = get_field('footer_social_buttons');
		}

		if(get_field('footer') && get_field('footer') == 'hide') {
			$style['footer'] = 'hide';
		}

		if(get_field('project_style') && get_field('project_style') != 'default') {
			$style['project_style'] = get_field('project_style');
		}

		if(get_field('project_image') && get_field('project_image') != 'default') {
			$style['project_image'] = get_field('project_image');
		}

		if(get_field('logo_variant') && get_field('logo_variant') != 'default') {
			$style['logo_variant'] = get_field('logo_variant');
		}
	}

	$style['css_classes'] = 'header_'.$style['header_style'];

	$style['css_classes'] .= ' '.$style['header_color_mode'];

	if($style['header_space'] == 'yes') {
		$style['css_classes'] .= ' header-space-on';
	}

	return $style;
}

add_filter( 'body_class', 'softa_custom_body_classes');
function softa_custom_body_classes( $classes ) {
	global $softa_theme;
	$site_mode = 'light';
	$header_style = 'default';

	if ( isset($softa_theme) && !empty($softa_theme) ) {

		if(isset($softa_theme['header_style']) && !empty($softa_theme['header_style'])) {
			$header_style = ' header_type_'.$softa_theme['header_style'];
		}
	}

	if(function_exists("get_field")) {

		if(get_field('header_style') && get_field('header_style') != 'default') {
			$header_style = ' header_type_'.get_field('header_style');
		}
	}

    $classes[] = $site_mode;
    $classes[] = $header_style;

    if(softa_styles()['right_click_disable'] == '1') {
		$classes[] = 'right-click-disable';
	}
 
    return $classes;
}

/**
 * Yprm Custom Head Script
 */

if (!function_exists('yprm_custom_head_script')) {
  function yprm_custom_head_script() {
    if (function_exists('softa_styles') && !empty(softa_styles()['code_in_head'])) {
      echo '<script>'.softa_styles()['code_in_head'].'</script>';
    }
  }
  add_action( 'wp_head', 'yprm_custom_head_script', 0 );
}

/**
 * Yprm Custom Footer Script
 */

if (!function_exists('yprm_custom_footer_script')) {
  function yprm_custom_footer_script() {
    if (function_exists('softa_styles') && !empty(softa_styles()['code_before_body'])) {
      echo '<script>'.softa_styles()['code_before_body'].'</script>';
    }
  }
  add_action( 'wp_footer', 'yprm_custom_footer_script', 500 );
}

add_action('wp_footer', 'softa_right_click_disable'); 
function softa_right_click_disable() {
	if(function_exists('softa_styles') && softa_styles()['right_click_disable'] == '1') {
	    echo '<div class="right-click-disable-message main-row"><div class="container"><div class="cell">'.wp_kses_post(softa_styles()['right_click_disable_message']).'</div></div></div>';
    } 
}

/**
 * Portfolio widget
 */
class softa_portfolio_widget extends WP_Widget {
 
	function __construct() {
		parent::__construct(
			'portfolio', 
			'Portfolio'
		);
	}
 
	public function widget( $args, $instance ) {
		$title = $instance['title'];
		$amount = $instance['amount'];
		$cols = $instance['cols'];
 
		switch ($cols) {
			case '1':
			$class = "col-xs-12";
			break;
			case '2':
			$class = "col-xs-6 col-sm-6";
			break;
			case '3':
			$class = "col-xs-4";
			break;
			case '4':
			$class = "col-xs-6 col-md-3";
			break;

			default:
			$class = "";
			break;
		}

		$porfolio_array = get_posts( array(
			'numberposts'     => $amount,
			'post_type'       => 'pt-portfolio',
			'post_status'     => 'publish'
			)
		);

		echo $args['before_widget'];
		if ( ! empty( $title ) ) echo $args['before_title'] . esc_html($title) . $args['after_title'];
		?>
		<div class="gallery-module row">
			<?php foreach ($porfolio_array as $item) {
				setup_postdata($item);
				$id = $item->ID;
				$name = $item->post_title;

				$thumb = get_post_meta( $id, '_thumbnail_id', true );

				$link = get_permalink($id);
			?>
			<div class="<?php echo esc_attr($class) ?> item"><a href="<?php echo esc_url($link) ?>"><?php echo wp_get_attachment_image( $thumb , 'thumbnail', true, array('title'=>$name) ) ?></a></div>
			<?php } ?>
		</div>
		<?php
		echo $args['after_widget'];
	}
 
	public function form( $instance ) {
		$title = "";
		if ( isset( $instance[ 'title' ] ) ) {
			$title = $instance[ 'title' ];
		}
		$amount = "";
		if ( isset( $instance[ 'amount' ] ) ) {
			$amount = $instance[ 'amount' ];
		}
		$cols = "";
		if ( isset( $instance[ 'cols' ] ) ) {
			$cols = $instance[ 'cols' ];
		}
		?>
		<p>
			<label for="<?php echo esc_html($this->get_field_id( 'title' )); ?>"><?php esc_html_e( 'Heading:', 'softa') ?></label> 
			<input class="widefat" id="<?php echo esc_attr($this->get_field_id( 'title' )); ?>" name="<?php echo esc_html($this->get_field_name( 'title' )); ?>" type="text" value="<?php echo esc_attr( $title ); ?>" />
		</p>
		<p>
			<label for="<?php echo esc_html($this->get_field_id( 'amount' )); ?>"><?php esc_html_e( 'Amount:', 'softa') ?></label> 
			<input id="<?php echo esc_html($this->get_field_id( 'amount' )); ?>" name="<?php echo esc_html($this->get_field_name( 'amount' )); ?>" type="text" value="<?php echo ($amount) ? esc_attr( $amount ) : '9'; ?>" size="3" />
		</p>
		<p>
			<label for="<?php echo esc_html($this->get_field_id( 'cols' )); ?>"><?php esc_html_e( 'Cols:', 'softa') ?></label> 
			<input id="<?php echo esc_html($this->get_field_id( 'cols' )); ?>" name="<?php echo esc_html($this->get_field_name( 'cols' )); ?>" type="text" value="<?php echo ($cols) ? esc_attr( $cols ) : '3'; ?>" size="3" />
		</p>
		<?php 
	}
 
	public function update( $new_instance, $old_instance ) {
		$instance = array();
		$instance['title'] = ( ! empty( $new_instance['title'] ) ) ? strip_tags( $new_instance['title'] ) : '';
		$instance['amount'] = ( is_numeric( $new_instance['amount'] ) ) ? $new_instance['amount'] : '8';
		$instance['cols'] = ( is_numeric( $new_instance['cols'] ) ) ? $new_instance['cols'] : '5';
		return $instance;
	}
}

function softa_portfolio_widget() {
	register_widget( 'softa_portfolio_widget' );
}
add_action( 'widgets_init', 'softa_portfolio_widget' );

function fb_mce_before_init( $settings ) {

    $style_formats = array(
        array(
            'title' => 'thin',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '100',
            )
        ),
        array(
            'title' => 'extra-light',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '200',
            )
        ),
        array(
            'title' => 'light',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '300',
            )
        ),
        array(
            'title' => 'regular',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '400',
            )
        ),
        array(
            'title' => 'medium',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '500',
            )
        ),
        array(
            'title' => 'semi-bold',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '600',
            )
        ),
        array(
            'title' => 'bold',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '700',
            )
        ),
        array(
            'title' => 'extra-bold',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '800',
            )
        ),
        array(
            'title' => 'black',
            'inline' => 'span',
            'styles' => array(
                'fontWeight'    => '900',
            )
        ),
        array(
            'title' => 'UPPERCASE',
            'inline' => 'span',
            'styles' => array(
                'textTransform'    => 'uppercase',
            )
        )
    );

    $settings['style_formats'] = json_encode( $style_formats );

    return $settings;
}
add_filter( 'tiny_mce_before_init', 'fb_mce_before_init' );

if(function_exists('vc_remove_param')) {
	vc_remove_param( 'vc_btn', 'color' );
}

if(function_exists('vc_add_param')) {
	vc_add_params( 'vc_custom_heading', array(
		array(
		    "type"        => "switch",
		    "class"       => "",
		    "heading"     => esc_html__( "Decor line", "softa" ),
		    "param_name"  => "decor_line",
		    "value"       => "off",
		    "options"     => array(
		        "on" => array(
		            "on"    => "On",
		            "off"   => "Off",
		        ),
		    ),
		    "dependency"  => "",
		    "default_set" => false,
		),
        array(
            "type"        => "dropdown",
            "heading"     => esc_html__( "Decor line style", "softa" ),
            "param_name"  => "decor_line_style",
            "value"      => array(
                esc_html__( "Style 1", "softa" ) => "style1",
                esc_html__( "Style 2", "softa" ) => "style2",
            ),
            "dependency" => Array("element" => "decor_line", "value" => "on" ),
        ),
        array(
            "type"        => "dropdown",
            "heading"     => esc_html__( "Decor line color", "softa" ),
            "param_name"  => "decor_line_bg",
            "value"      => array(
                esc_html__( "Default", "softa" ) => "default",
                esc_html__( "Color", "softa" ) => "color",
                esc_html__( "Gradient", "softa" ) => "gradient",
            ),
            "dependency" => Array("element" => "decor_line_style", "value" => "style1" ),
        ),
        array(
            "type" => "colorpicker",
            "heading" => esc_html__("Decor line color", "softa"),
            "param_name" => "decor_line_bg_color",
            "dependency" => Array("element" => "decor_line_bg", "value" => "color" ),
        ),
        array(
            "type" => "gradient",
            "base_gradient" => "#8c80ce 0%,#e04391 100%",
            "base_orientation" => "horizontal",
            "heading" => esc_html__("Decor line gradient", "softa"),
            "param_name" => "decor_line_bg_gradient",
            "dependency" => Array("element" => "decor_line_bg", "value" => "gradient" ),
        ),
	));

	vc_add_params("vc_row", array(
		array(
			'type' => 'colorpicker',
			'heading' => __( 'Custom text color', 'softa' ),
			'param_name' => 'custom_text_color',
		),
		array(
			"type" => "dropdown",
			"class" => "",
			"heading" => esc_html__( "Background position", "softa" ),
			"param_name" => "background_position",
			"value" => array(
				'' => 'None',
				"center center" => "center center",
				"center top" => "center top",
				"center bottom" => "center bottom"
			)
		),
        array(
			'type' => 'checkbox',
			'heading' => __( 'Gradient background', 'softa' ),
			'param_name' => 'show_background_gradient',
		),
		array(
			'type' => 'gradient',
			'heading' => __( 'Gradient', 'softa' ),
			'param_name' => 'background_gradient',
			'dependency' => array(
				'element' => 'show_background_gradient',
				'not_empty' => true,
			),
		),
        array(
			'type' => 'checkbox',
			'heading' => __( 'Сolor overlay', 'softa' ),
			'param_name' => 'color_overlay',
		),
		array(
			'type' => 'colorpicker',
			'heading' => __( 'Сolor overlay hex', 'softa' ),
			'param_name' => 'color_overlay_color',
			'dependency' => array(
				'element' => 'color_overlay',
				'not_empty' => true,
			),
		),
	));

	vc_remove_param( "vc_icon", "background_color" );
	vc_remove_param( "vc_icon", "custom_color" );
	vc_remove_param( "vc_icon", "background_style" );
	vc_remove_param( "vc_icon", "background_color" );
	vc_remove_param( "vc_icon", "custom_background_color" );
	vc_remove_param( "vc_icon", "size" );
	vc_remove_param( "vc_icon", "align" );
	vc_remove_param( "vc_icon", "link" );
	vc_remove_param( "vc_icon", "el_id" );
	vc_remove_param( "vc_icon", "el_class" );
	vc_remove_param( "vc_icon", "css" );

  if(function_exists('vc_get_shared')) {
    vc_add_params("vc_icon", array(
      array(
        'type' => 'colorpicker',
        'heading' => __( 'Custom color', 'softa' ),
        'param_name' => 'custom_color',
        'description' => __( 'Select custom icon color.', 'softa' ),
        'dependency' => array(
          'element' => 'color',
          'value' => 'custom',
        ),
      ),
      array(
        'type' => 'dropdown',
        'heading' => __( 'Background shape', 'softa' ),
        'param_name' => 'background_style',
        'value' => array(
          __( 'None', 'softa' ) => '',
          __( 'Circle', 'softa' ) => 'rounded',
          __( 'Square', 'softa' ) => 'boxed',
          __( 'Rounded', 'softa' ) => 'rounded-less',
          __( 'Outline Circle', 'softa' ) => 'rounded-outline',
          __( 'Outline Square', 'softa' ) => 'boxed-outline',
          __( 'Outline Rounded', 'softa' ) => 'rounded-less-outline',
        ),
        'description' => __( 'Select background shape and style for icon.', 'softa' ),
      ),
      array(
        'type' => 'dropdown',
        'heading' => esc_html__( 'Background color', 'js_composer' ),
        'param_name' => 'background_color',
        'value' => array_merge( vc_get_shared( 'colors' ), array( esc_html__( 'Custom color', 'js_composer' ) => 'custom', esc_html__( 'Gradient', 'js_composer' ) => 'gradient' ) ),
        'std' => 'grey',
        'description' => esc_html__( 'Select background color for icon.', 'js_composer' ),
        'param_holder_class' => 'vc_colored-dropdown',
        'dependency' => array(
          'element' => 'background_style',
          'not_empty' => true,
        ),
      ),
      array(
        'type' => 'colorpicker',
        'heading' => __( 'Custom background color', 'softa' ),
        'param_name' => 'custom_background_color',
        'description' => __( 'Select custom icon background color.', 'softa' ),
        'dependency' => array(
          'element' => 'background_color',
          'value' => 'custom',
        ),
      ),
      array(
        'type' => 'gradient',
        'heading' => __( 'Custom background gradient', 'softa' ),
        'param_name' => 'custom_background_gradient',
        'dependency' => array(
          'element' => 'background_color',
          'value' => 'gradient',
        ),
      ),
      array(
        'type' => 'dropdown',
        'heading' => __( 'Size', 'softa' ),
				'param_name' => 'size',
				'value' => array_merge( vc_get_shared( 'sizes' ), array( 'Extra Large' => 'xl' ) ),
        'std' => 'md',
        'description' => __( 'Icon size.', 'softa' ),
      ),
      array(
        'type' => 'dropdown',
        'heading' => __( 'Icon alignment', 'softa' ),
        'param_name' => 'align',
        'value' => array(
          __( 'Left', 'softa' ) => 'left',
          __( 'Right', 'softa' ) => 'right',
          __( 'Center', 'softa' ) => 'center',
        ),
        'description' => __( 'Select icon alignment.', 'softa' ),
      ),
      array(
        'type' => 'vc_link',
        'heading' => __( 'URL (Link)', 'softa' ),
        'param_name' => 'link',
        'description' => __( 'Add link to icon.', 'softa' ),
      ),
      vc_map_add_css_animation(),
      array(
        'type' => 'el_id',
        'heading' => __( 'Element ID', 'softa' ),
        'param_name' => 'el_id',
        'description' => sprintf( __( 'Enter element ID (Note: make sure it is unique and valid according to <a href="%s" target="_blank">w3c specification</a>).', 'softa' ), 'http://www.w3schools.com/tags/att_global_id.asp' ),
      ),
      array(
        'type' => 'textfield',
        'heading' => __( 'Extra class name', 'softa' ),
        'param_name' => 'el_class',
        'description' => __( 'Style particular content element differently - add a class name and refer to it in custom CSS.', 'softa' ),
      ),
      array(
        'type' => 'css_editor',
        'heading' => __( 'CSS box', 'softa' ),
        'param_name' => 'css',
        'group' => __( 'Design Options', 'softa' ),
      ),
    ));
  }

	vc_remove_param( "vc_tta_accordion", "no_fill" );
	vc_remove_param( "vc_tta_accordion", "spacing" );
	vc_remove_param( "vc_tta_accordion", "gap" );
	vc_remove_param( "vc_tta_accordion", "c_align" );
	vc_remove_param( "vc_tta_accordion", "autoplay" );
	vc_remove_param( "vc_tta_accordion", "collapsible_all" );
	vc_remove_param( "vc_tta_accordion", "c_icon" );
	vc_remove_param( "vc_tta_accordion", "c_position" );
	vc_remove_param( "vc_tta_accordion", "active_section" );
	vc_remove_param( "vc_tta_accordion", "el_id" );
	vc_remove_param( "vc_tta_accordion", "el_class" );
	vc_remove_param( "vc_tta_accordion", "css" );

	vc_add_params("vc_tta_accordion", array(
        array(
            "type" => "gradient",
            "base_gradient" => "#ff6884 0%,#620044 100%",
            "base_orientation" => "horizontal",
            "heading" => esc_html__("Gradient on active", "softa"),
            "param_name" => "active_gradient",
			'dependency' => array(
				'element' => 'style',
				'value' => 'classic',
			),
        ),
        array(
            "type" => "colorpicker",
            "heading" => esc_html__("Text color active block", "softa"),
            "param_name" => "active_color",
			'dependency' => array(
				'element' => 'style',
				'value' => 'classic',
			),
        ),
		array(
			'type' => 'checkbox',
			'param_name' => 'no_fill',
			'heading' => __( 'Do not fill content area?', 'softa' ),
			'description' => __( 'Do not fill content area with color.', 'softa' ),
		),
		array(
			'type' => 'dropdown',
			'param_name' => 'spacing',
			'value' => array(
				__( 'None', 'softa' ) => '',
				'1px' => '1',
				'2px' => '2',
				'3px' => '3',
				'4px' => '4',
				'5px' => '5',
				'10px' => '10',
				'15px' => '15',
				'20px' => '20',
				'25px' => '25',
				'30px' => '30',
				'35px' => '35',
			),
			'heading' => __( 'Spacing', 'softa' ),
			'description' => __( 'Select accordion spacing.', 'softa' ),
		),
		array(
			'type' => 'dropdown',
			'param_name' => 'gap',
			'value' => array(
				__( 'None', 'softa' ) => '',
				'1px' => '1',
				'2px' => '2',
				'3px' => '3',
				'4px' => '4',
				'5px' => '5',
				'10px' => '10',
				'15px' => '15',
				'20px' => '20',
				'25px' => '25',
				'30px' => '30',
				'35px' => '35',
			),
			'heading' => __( 'Gap', 'softa' ),
			'description' => __( 'Select accordion gap.', 'softa' ),
		),
		array(
			'type' => 'dropdown',
			'param_name' => 'c_align',
			'value' => array(
				__( 'Left', 'softa' ) => 'left',
				__( 'Right', 'softa' ) => 'right',
				__( 'Center', 'softa' ) => 'center',
			),
			'heading' => __( 'Alignment', 'softa' ),
			'description' => __( 'Select accordion section title alignment.', 'softa' ),
		),
		array(
			'type' => 'dropdown',
			'param_name' => 'autoplay',
			'value' => array(
				__( 'None', 'softa' ) => 'none',
				'1' => '1',
				'2' => '2',
				'3' => '3',
				'4' => '4',
				'5' => '5',
				'10' => '10',
				'20' => '20',
				'30' => '30',
				'40' => '40',
				'50' => '50',
				'60' => '60',
			),
			'std' => 'none',
			'heading' => __( 'Autoplay', 'softa' ),
			'description' => __( 'Select auto rotate for accordion in seconds (Note: disabled by default).', 'softa' ),
		),
		array(
			'type' => 'checkbox',
			'param_name' => 'collapsible_all',
			'heading' => __( 'Allow collapse all?', 'softa' ),
			'description' => __( 'Allow collapse all accordion sections.', 'softa' ),
		),
		// Control Icons
		array(
			'type' => 'dropdown',
			'param_name' => 'c_icon',
			'value' => array(
				__( 'None', 'softa' ) => '',
				__( 'Chevron', 'softa' ) => 'chevron',
				__( 'Plus', 'softa' ) => 'plus',
				__( 'Triangle', 'softa' ) => 'triangle',
			),
			'std' => 'plus',
			'heading' => __( 'Icon', 'softa' ),
			'description' => __( 'Select accordion navigation icon.', 'softa' ),
		),
		array(
			'type' => 'dropdown',
			'param_name' => 'c_position',
			'value' => array(
				__( 'Left', 'softa' ) => 'left',
				__( 'Right', 'softa' ) => 'right',
			),
			'dependency' => array(
				'element' => 'c_icon',
				'not_empty' => true,
			),
			'heading' => __( 'Position', 'softa' ),
			'description' => __( 'Select accordion navigation icon position.', 'softa' ),
		),
		// Control Icons END
		array(
			'type' => 'textfield',
			'param_name' => 'active_section',
			'heading' => __( 'Active section', 'softa' ),
			'value' => 1,
			'description' => __( 'Enter active section number (Note: to have all sections closed on initial load enter non-existing number).', 'softa' ),
		),
		vc_map_add_css_animation(),
		array(
			'type' => 'el_id',
			'heading' => __( 'Element ID', 'softa' ),
			'param_name' => 'el_id',
			'description' => sprintf( __( 'Enter element ID (Note: make sure it is unique and valid according to <a href="%s" target="_blank">w3c specification</a>).', 'softa' ), 'http://www.w3schools.com/tags/att_global_id.asp' ),
		),
		array(
			'type' => 'textfield',
			'heading' => __( 'Extra class name', 'softa' ),
			'param_name' => 'el_class',
			'description' => __( 'If you wish to style particular content element differently, then use this field to add a class name and then refer to it in your css file.', 'softa' ),
		),
		array(
			'type' => 'css_editor',
			'heading' => __( 'CSS box', 'softa' ),
			'param_name' => 'css',
			'group' => __( 'Design Options', 'softa' ),
		),
	));
}

add_action( 'vc_before_init', 'vc_before_init_actions' );

function vc_before_init_actions() {
    if( function_exists('vc_set_shortcodes_templates_dir') ){
        vc_set_shortcodes_templates_dir( get_template_directory() . '/vc_templates' );
    }
}

add_filter( 'jpeg_quality', 'softa_filter_theme_image_full_quality' );
add_filter( 'wp_editor_set_quality', 'softa_filter_theme_image_full_quality' );

function softa_filter_theme_image_full_quality( $quality ) {
    return 100;
}

class Child_Wrap extends Walker_Nav_Menu {
    function start_lvl(&$output, $depth = 0, $args = array()) {
        $indent = str_repeat("\t", $depth);
        $output .= "\n$indent<ul class=\"sub-menu\"><li class=\"back multimedia-icon-back\"></li>\n";
    }
    function end_lvl(&$output, $depth = 0, $args = array()) {
        $indent = str_repeat("\t", $depth);
        $output .= "$indent</ul>\n";
    }
}

add_action( 'vc_before_init', 'softa_vcSetAsTheme' );
function softa_vcSetAsTheme() {
	if(function_exists('vc_set_as_theme')) {
    	vc_set_as_theme();
	}
}

remove_action( 'woocommerce_before_shop_loop_item_title', 'woocommerce_template_loop_product_thumbnail', 10);
add_action( 'woocommerce_before_shop_loop_item_title', 'softa_woocommerce_template_loop_product_thumbnail', 10);

/**
 * WooCommerce Loop Product Thumbs
 **/
if ( ! function_exists( 'softa_woocommerce_template_loop_product_thumbnail' ) ) {
	function softa_woocommerce_template_loop_product_thumbnail() {
		echo softa_woocommerce_get_product_thumbnail();
	} 
}

/**
 * WooCommerce Product Thumbnail
 **/
if ( ! function_exists( 'softa_woocommerce_get_product_thumbnail' ) ) {
	
	function softa_woocommerce_get_product_thumbnail( $size = 'shop_catalog', $placeholder_width = 0, $placeholder_height = 0  ) {
		global $post, $woocommerce,$product;
		if ( ! $placeholder_width )
			$placeholder_width = wc_get_image_size( 'shop_catalog' )['width'];
		if ( ! $placeholder_height )
			$placeholder_height = wc_get_image_size( 'shop_catalog' )['height'];

		$output = '<div class="image">';
		$output .= '<a href="'.get_the_permalink().'" class="img">';
			if ( has_post_thumbnail() ) {

				$output .= get_the_post_thumbnail( $post->ID, $size );
				if($attachment_ids = $product->get_gallery_image_ids() ) {
					if(isset($attachment_ids[1])){
						$output .= wp_get_attachment_image( $attachment_ids[1] , 'shop_catalog', '', array('class'=>'show') );
					}
				}

			} else {
				
				$output .= '<img src="'. woocommerce_placeholder_img_src() .'" alt="Placeholder" width="' . $placeholder_width . '" height="' . $placeholder_height . '" />';
				
			}

		$output .= '</a></div>';

		return $output;
	}
}

/**
 * WooCommerce Single Meta
 **/
remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_meta', 40 );
add_action( 'woocommerce_single_product_summary', 'softa_woocommerce_template_single_meta_remove_category', 5 );

function softa_woocommerce_template_single_meta_remove_category(){    

	global $product;

	?>
	<div class="product_meta">

	  <?php do_action( 'woocommerce_product_meta_start' ); ?>

	  <?php if ( wc_product_sku_enabled() && $product->get_sku() && ( $product->get_sku() || $product->is_type( 'variable' ) ) ) : ?>

	    <span class="sku_wrapper"><?php _e( 'SKU:', 'softa' ); ?> <span class="sku" itemprop="sku"><?php echo ( $sku = $product->get_sku() ) ? $sku : __( '---', 'softa' ); ?></span></span>

	  <?php endif; ?>

	 
	  <?php do_action( 'woocommerce_product_meta_end' ); ?>

	</div>

<?php }

function softa_related_products_limit() {
    global $product;
    $orderby = '';
    $columns = 4;
    $related = $product->get_related( 4 );
    $args = array(
        'post_type'           => 'product',
        'no_found_rows'       => 1,
        'posts_per_page'      => 4,
        'ignore_sticky_posts' => 1,
        'orderby'             => $orderby,
        'post__in'            => $related,
        'post__not_in'        => array($product->get_id())
    );
    return $args;
}
add_filter( 'woocommerce_related_products_args', 'softa_related_products_limit' );


/**
 * The order of the blocks
 **/

remove_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_price', 10);

add_filter('woocommerce_add_to_cart_fragments', 'woocommerce_header_add_to_cart_fragment');

function woocommerce_header_add_to_cart_fragment( $fragments ) {
	global $woocommerce;

	ob_start();
	?>
	<div class="header-minicart woocommerce header-minicart-softa">
		<?php $count = $woocommerce->cart->cart_contents_count;
		if($count == 0) { ?>
		<div class="hm-cunt"><i class="basic-ui-icon-shopping-cart"></i><span><?php echo esc_html($count) ?></span></div>
		<?php } else { ?>
		<a class="hm-cunt" href="<?php echo esc_url(wc_get_cart_url()) ?>"><i class="basic-ui-icon-shopping-cart"></i><span><?php echo esc_html($count) ?></span></a>
		<?php } ?>
		<div class="minicart-wrap">
			<?php woocommerce_mini_cart(); ?>
		</div>
	</div>
	<?php
	$fragments['.header-minicart-softa'] = ob_get_clean();

	return $fragments;
}

function is_realy_woocommerce_page() {
	if(  function_exists ( "is_woocommerce" ) && is_woocommerce()){
		return true;
	}
	$woocommerce_keys   =   array ( "woocommerce_shop_page_id" ,
		"woocommerce_terms_page_id" ,
		"woocommerce_cart_page_id" ,
		"woocommerce_checkout_page_id" ,
		"woocommerce_pay_page_id" ,
		"woocommerce_thanks_page_id" ,
		"woocommerce_myaccount_page_id" ,
		"woocommerce_edit_address_page_id" ,
		"woocommerce_view_order_page_id" ,
		"woocommerce_change_password_page_id" ,
		"woocommerce_logout_page_id" ,
		"woocommerce_lost_password_page_id" ) ;
	foreach ( $woocommerce_keys as $wc_page_id ) {
		if ( get_the_ID () == get_option ( $wc_page_id , 0 ) ) {
			return true ;
		}
	}
	return false;
}

function softa_phpinfo2array() {
    $entitiesToUtf8 = function($input) {
        // http://php.net/manual/en/function.html-entity-decode.php#104617
        return preg_replace_callback("/(&#[0-9]+;)/", function($m) { return mb_convert_encoding($m[1], "UTF-8", "HTML-ENTITIES"); }, $input);
    };
    $plainText = function($input) use ($entitiesToUtf8) {
        return trim(html_entity_decode($entitiesToUtf8(strip_tags($input))));
    };
    $titlePlainText = function($input) use ($plainText) {
        return '# '.$plainText($input);
    };
    
    ob_start();
    phpinfo(-1);
    
    $phpinfo = array('phpinfo' => array());

    // Strip everything after the <h1>Configuration</h1> tag (other h1's)
    if (!preg_match('#(.*<h1[^>]*>\s*Configuration.*)<h1#s', ob_get_clean(), $matches)) {
        return array();
    }
    
    $input = $matches[1];
    $matches = array();

    if(preg_match_all(
        '#(?:<h2.*?>(?:<a.*?>)?(.*?)(?:<\/a>)?<\/h2>)|'.
        '(?:<tr.*?><t[hd].*?>(.*?)\s*</t[hd]>(?:<t[hd].*?>(.*?)\s*</t[hd]>(?:<t[hd].*?>(.*?)\s*</t[hd]>)?)?</tr>)#s',
        $input, 
        $matches, 
        PREG_SET_ORDER
    )) {
        foreach ($matches as $match) {
            $fn = strpos($match[0], '<th') === false ? $plainText : $titlePlainText;
            if (strlen($match[1])) {
                $phpinfo[$match[1]] = array();
            } elseif (isset($match[3])) {
                $keys1 = array_keys($phpinfo);
                $phpinfo[end($keys1)][$fn($match[2])] = isset($match[4]) ? array($fn($match[3]), $fn($match[4])) : $fn($match[3]);
            } else {
                $keys1 = array_keys($phpinfo);
                $phpinfo[end($keys1)][] = $fn($match[2]);
            }

        }
    }
    
    return $phpinfo;
}

function softa_let_to_num( $size ) {
	$l    = substr( $size, -1 );
	$ret  = substr( $size, 0, -1 );
	$byte = 1024;

	switch ( strtoupper( $l ) ) {
		case 'P':
			$ret *= 1024;
		case 'T':
			$ret *= 1024;
		case 'G':
			$ret *= 1024;
		case 'M':
			$ret *= 1024;
		case 'K':
			$ret *= 1024;
	}
	return $ret;
}

add_filter( 'the_password_form', 'softa_custom_password_form' );
function softa_custom_password_form() {
    global $post;
    $label = 'pwbox-'.( empty( $post->ID ) ? rand() : $post->ID );
    $o = '<form class="protected-post-form" action="' . esc_url( site_url( 'wp-login.php?action=postpass', 'login_post' ) ) . '" method="post"><div class="cell">'. wp_kses_post(softa_styles()['protected_message']) .'<div class="area"><input name="post_password" placeholder="'.esc_attr__('Type the password', 'softa').'" id="' . $label . '" type="password" /><button type="submit" name="Submit" class="button"><i class="basic-ui-icon-unlock"></i></button></div></div></form>';
    return $o;
}

/**
 * FIX Unyson curl issue
 */
if ( ! function_exists( '_mwt_fw_filter_github_api_url' ) ) :
	function _mwt_fw_filter_github_api_url( $url ) {
		return 'https://api.github.com';
	}
endif;

add_filter( 'fw_github_api_url', '_mwt_fw_filter_github_api_url', 999 );
