<?php

if (!defined('ABSPATH')) {
    exit;
}

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Element_Base;
use Elementor\Core\Files\CSS;

class Razox_Elementor_Custom_Css {
    public function __construct() {
        add_action('elementor/init', [$this, 'on_elementor_init']);
    }

    public function on_elementor_init() {
        add_action('elementor/element/after_section_end', [$this, 'register_controls'], 10, 2);
        add_action('elementor/element/parse_css', [$this, 'add_post_css'], 10, 2);
        add_action('elementor/css-file/post/parse', [$this, 'add_page_settings_css']);
        add_action('elementor/editor/before_enqueue_scripts', [$this, 'add_scripts_editor']);
        add_action('elementor/element/container/section_layout_additional_options/after_section_end', function ($container) {
            $container->start_controls_section(
                'razox_section_effect',
                [
                    'label' => esc_html__('Custom', 'razox'),
                    'tab'   => Controls_Manager::TAB_LAYOUT,
                ]
            );

            $container->add_control(
                'effect_moving_image',
                [
                    'label'        => esc_html__('Effect Parallax Image', 'razox'),
                    'type'         => Controls_Manager::SWITCHER,
                    'default'      => '',
                    'prefix_class' => 'razox-effect-moving-image-',
                ]
            );
            $container->add_control(
                'razox_change_scrollup',
                [
                    'label'        => esc_html__('Change Scrollup Color', 'razox'),
                    'type'         => Controls_Manager::SWITCHER,
                    'default'      => '',
                    'prefix_class' => 'razox-scrollup-change-color-',
                ]
            );

            $container->add_control(
                'razox_color_scrollup',
                [
                    'label'     => esc_html__('Background Active', 'razox'),
                    'type'      => Controls_Manager::COLOR,
                    'selectors' => [
                        '.scrollup.change-color[data-id="{{ID}}"]'                        => 'color: {{VALUE}}',
                        '.scrollup.change-color[data-id="{{ID}}"] .scrollup-icon::before' => 'color: {{VALUE}}',
                    ],
                    'condition' => [
                        'razox_change_scrollup' => 'yes'
                    ]
                ]
            );

            $container->end_controls_section();
        });
        add_action('elementor/element/container/section_shape_divider/after_section_start', function ($container) {
            $container->add_control(
                'custom_shape_divider',
                [
                    'label'        => esc_html__('Shape Divider Theme', 'razox'),
                    'type'         => Controls_Manager::SWITCHER,
                    'default'      => '',
                    'prefix_class' => 'custom_shape_divider-'
                ]
            );

        });

        add_action('elementor/element/section/section_shape_divider/after_section_start', function ($container) {
            $container->add_control(
                'custom_shape_divider',
                [
                    'label'        => esc_html__('Shape Divider Theme', 'razox'),
                    'type'         => Controls_Manager::SWITCHER,
                    'default'      => '',
                    'prefix_class' => 'custom_shape_divider-',
                ]
            );

        });

    }

    public function add_scripts_editor() {
        wp_enqueue_script('razox-elementor-custom-css', get_theme_file_uri('/assets/js/elementor-custom-css.js'), [], RAZOX_VERSION, true);
    }

    /**
     * @param $post_css CSS\Post
     * @param $element Element_Base
     */
    public function add_post_css($post_css, $element) {
        $element_settings = $element->get_settings();

        if (empty($element_settings['custom_css'])) {
            return;
        }

        $css = trim($element_settings['custom_css']);

        if (empty($css)) {
            return;
        }
        $css = str_replace('selector', $post_css->get_element_unique_selector($element), $css);

        // Add a css comment
        $css = sprintf('/* Start custom CSS for %s, class: %s */', $element->get_name(), $element->get_unique_selector()) . $css . '/* End custom CSS */';

        $post_css->get_stylesheet()->add_raw_css($css);
    }

    /**
     * @param $element Controls_Stack
     * @param $section_id string
     */
    public function register_controls(Controls_Stack $element, $section_id) {
        // Remove Custom CSS Banner (From free version)
        if ('section_custom_css_pro' !== $section_id) {
            return;
        }

        $old_section        = \Elementor\Plugin::instance()->controls_manager->get_control_from_stack($element->get_unique_name(), 'section_custom_css_pro');
        $controls_to_remove = ['section_custom_css_pro', 'custom_css_pro'];

        \Elementor\Plugin::instance()->controls_manager->remove_control_from_stack($element->get_unique_name(), $controls_to_remove);

        $element->start_controls_section(
            'section_custom_css',
            [
                'label' => esc_html__('Custom CSS', 'razox'),
                'tab'   => $old_section['tab'],
            ]
        );

        $element->add_control(
            'custom_css_title',
            [
                'raw'  => esc_html__('Add your own custom CSS here', 'razox'),
                'type' => Controls_Manager::RAW_HTML,
            ]
        );

        $element->add_control(
            'custom_css',
            [
                'type'        => Controls_Manager::CODE,
                'label'       => esc_html__('Custom CSS', 'razox'),
                'language'    => 'css',
                'render_type' => 'ui',
                'show_label'  => false,
                'separator'   => 'none',
            ]
        );

        $element->add_control(
            'custom_css_description',
            [
                'raw'             => wp_kses_post(__('Use "selector" to target wrapper element. Examples:<br>selector {color: red;} // For main element<br>selector .child-element {margin: 10px;} // For child element<br>.my-class {text-align: center;} // Or use any custom selector', 'razox')),
                'type'            => Controls_Manager::RAW_HTML,
                'content_classes' => 'elementor-descriptor',
            ]
        );

        $element->end_controls_section();
    }

    /**
     * @param $post_css CSS\Post
     */
    public function add_page_settings_css($post_css) {
        $document   = \Elementor\Plugin::instance()->documents->get($post_css->get_post_id());
        $custom_css = $document->get_settings('custom_css');

        $custom_css = trim($custom_css);

        if (empty($custom_css)) {
            return;
        }

        $custom_css = str_replace('selector', 'body.elementor-page-' . $post_css->get_post_id(), $custom_css);

        // Add a css comment
        $custom_css = '/* Start custom CSS for page-settings */' . $custom_css . '/* End custom CSS */';

        $post_css->get_stylesheet()->add_raw_css($custom_css);
    }

    /**
     * @param $element Element_Base
     */
    public function remove_go_pro_custom_css($element) {
        $controls_to_remove = ['section_custom_css_pro', 'custom_css_pro'];

        \Elementor\Plugin::instance()->controls_manager->remove_control_from_stack($element->get_unique_name(), $controls_to_remove);
    }
}

return new Razox_Elementor_Custom_Css();
