<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}
if (!razox_is_woocommerce_activated()) {
    return;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Icons_Manager;
use Elementor\Repeater;
use Razox\Elementor\Razox_Base_Widgets;

/**
 * Elementor Razox_Elementor_Products_Categories
 * @since 1.0.0
 */
class Razox_Elementor_Products_Categories extends Razox_Base_Widgets {

    public function get_categories() {
        return array('razox-addons');
    }

    /**
     * Get widget name.
     *
     * Retrieve tabs widget name.
     *
     * @return string Widget name.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_name() {
        return 'razox-product-categories';
    }

    /**
     * Get widget title.
     *
     * Retrieve tabs widget title.
     *
     * @return string Widget title.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_title() {
        return esc_html__('Razox Product Categories', 'razox');
    }

    /**
     * Get widget icon.
     *
     * Retrieve tabs widget icon.
     *
     * @return string Widget icon.
     * @since  1.0.0
     * @access public
     *
     */
    public function get_icon() {
        return 'eicon-tabs';
    }

    public function get_script_depends() {
        return ['razox-elementor-product-categories'];
    }

    public function get_style_depends(): array {
        return [ 'e-swiper' ];
    }

    public function on_export($element) {
        unset($element['settings']['category_image']['id']);

        return $element;
    }

    protected function register_controls() {

        //Section Query
        $this->start_controls_section(
            'section_setting',
            [
                'label' => esc_html__('Categories', 'razox'),
                'tab'   => Controls_Manager::TAB_CONTENT,
            ]
        );
        $repeater = new Repeater();

        $repeater->add_control(
            'categories_name',
            [
                'label' => esc_html__('Alternate Name', 'razox'),
                'type'  => Controls_Manager::TEXT,
            ]
        );

        $repeater->add_control(
            'categories_sub_name',
            [
                'label' => esc_html__('Sub', 'razox'),
                'type'  => Controls_Manager::TEXT,
            ]
        );

        $repeater->add_control(
            'categories',
            [
                'label'       => esc_html__('Categories', 'razox'),
                'type'        => Controls_Manager::SELECT2,
                'label_block' => true,
                'options'     => $this->get_product_categories(),
                'multiple'    => false,
            ]
        );

        $repeater->add_control(
            'cat-select',
            [
                'label'   => esc_html__('Select', 'razox'),
                'type'    => Controls_Manager::SELECT,
                'options' => [
                    'image' => __('Image', 'razox'),
                    'icon'  => __('Icon', 'razox'),
                ],
                'default' => 'image',

            ]
        );

        $repeater->add_control(
            'selected_icon',
            [
                'label'     => esc_html__('Icon', 'razox'),
                'type'      => Controls_Manager::ICONS,
                'condition' => ['cat-select' => 'icon'],
            ]
        );

        $repeater->add_control(
            'category_image',
            [
                'label'      => esc_html__('Choose Image', 'razox'),
                'default'    => [
                    'url' => Elementor\Utils::get_placeholder_image_src(),
                ],
                'type'       => Controls_Manager::MEDIA,
                'show_label' => false,
                'condition'  => ['cat-select' => 'image'],
            ]
        );

        $repeater->add_control(
            'display_background',
            [
                'label'        => esc_html__('Display Background', 'razox'),
                'type'         => \Elementor\Controls_Manager::SWITCHER,
                'label_on'     => esc_html__('Yes', 'razox'),
                'label_off'    => esc_html__('No', 'razox'),
                'return_value' => 'yes',
                'default'      => 'no',
            ]
        );

        $repeater->add_control(
            'category_background',
            [
                'label'      => esc_html__('Choose Background', 'razox'),
                'default'    => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
                'type'       => Controls_Manager::MEDIA,
                'show_label' => true,
                'condition'  => [
                    'display_background' => 'yes',
                ],
            ]
        );

        $this->add_control(
            'categories_list',
            [
                'label'       => esc_html__('Items', 'razox'),
                'type'        => Controls_Manager::REPEATER,
                'fields'      => $repeater->get_controls(),
                'title_field' => '{{{ categories }}}',
            ]
        );

        $this->add_group_control(
            Elementor\Group_Control_Image_Size::get_type(),
            [
                'name'      => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `brand_image_size` and `brand_image_custom_dimension`.
                'default'   => 'full',
                'separator' => 'none',
            ]
        );


        $this->add_control(
            'style',
            [
                'label'        => esc_html__('Style', 'razox'),
                'type'         => Controls_Manager::SELECT,
                'default'      => '1',
                'options'      => [
                    '1' => esc_html__('Style 1', 'razox'),
                    '2' => esc_html__('Style 2', 'razox'),
                    '3' => esc_html__('Style 3', 'razox'),
                    '4' => esc_html__('Style 4', 'razox'),
                ],
                'prefix_class' => 'category-product-style-'
            ]
        );

        $this->add_control('show_count',
            [
                'label' => esc_html__('Show Count', 'razox'),
                'type'  => Controls_Manager::SWITCHER,
            ]
        );

        $this->add_control(
            'button_text',
            [
                'label'       => esc_html__('Button Text', 'razox'),
                'type'        => Controls_Manager::TEXT,
                'default'     => esc_html__('View products', 'razox'),
                'placeholder' => esc_html__('View products', 'razox'),
                'label_block' => true,
                'condition'   => ['style' => '1'],
            ]
        );


        $this->add_responsive_control(
            'alignment',
            [
                'label'     => esc_html__('Alignment', 'razox'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'left'   => [
                        'title' => esc_html__('Left', 'razox'),
                        'icon'  => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__('Center', 'razox'),
                        'icon'  => 'eicon-text-align-center',
                    ],
                    'right'  => [
                        'title' => esc_html__('Right', 'razox'),
                        'icon'  => 'eicon-text-align-right',
                    ],
                ],
                'default'   => 'center',
                'selectors' => [
                    '{{WRAPPER}} .elementor-widget-container' => 'text-align: {{VALUE}}',
                    '{{WRAPPER}} .category-product-caption'   => 'text-align: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'box_section_style',
            [
                'label' => esc_html__('Box', 'razox'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'box_padding',
            [
                'label'      => esc_html__('Padding', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} div .product-cat' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'box_margin',
            [
                'label'      => esc_html__('Margin', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );

        $this->add_responsive_control(
            'min-height',
            [
                'label'      => esc_html__('Height', 'razox'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 100,
                        'max' => 1000,
                    ],
                    'vh' => [
                        'min' => 10,
                        'max' => 100,
                    ],
                ],
                'size_units' => ['px', 'vh'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'min-height: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->add_control(
            'box_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .product-cat' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tabs_box_style');
        $this->start_controls_tab(
            'box_img_normal',
            [
                'label' => esc_html__('Normal', 'razox'),
            ]
        );


        $this->add_control(
            'box_background',
            [
                'label'     => esc_html__('Background Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .product-cat' => 'background-color: {{VALUE}};',
                ],
            ]
        );


        $this->add_control(
            'box_opacity',
            [
                'label'     => esc_html__('Opacity', 'razox'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max'  => 1,
                        'min'  => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .product-cat' => 'opacity: {{SIZE}};',
                ],
            ]
        );
        $this->end_controls_tab();
        $this->start_controls_tab(
            'tab_box_hover',
            [
                'label' => esc_html__('Hover', 'razox'),
            ]
        );


        $this->add_control(
            'box_background_hover',
            [
                'label'     => esc_html__('Background Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->add_control(
            'box_opacity_hover',
            [
                'label'     => esc_html__('Opacity', 'razox'),
                'type'      => Controls_Manager::SLIDER,
                'range'     => [
                    'px' => [
                        'max'  => 1,
                        'min'  => 0.10,
                        'step' => 0.01,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover' => 'opacity: {{SIZE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->add_responsive_control(
            'effects_width',
            [
                'label'          => esc_html__('Width Effects', 'razox'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'size_units'     => ['%', 'px', 'vw'],
                'range'          => [
                    '%'  => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .category-product-caption' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'image_style',
            [
                'label' => esc_html__('Image', 'razox'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'image_background',
            [
                'label'     => esc_html__('Background Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .category-product-img' => 'background-color: {{VALUE}};',
                ],
                'condition'   => ['style' => '2'],
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'      => 'image_border',
                'selector'  => '{{WRAPPER}} .category-product-img',
                'separator' => 'before',
            ]
        );
        $this->add_responsive_control(
            'image_border_radius',
            [
                'label'      => esc_html__('Border radius', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .category-product-img, {{WRAPPER}} .category-product-img img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                    '{{WRAPPER}} .category-product-img a'                                      => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}',
                ],
            ]
        );
        $this->add_group_control(
            Group_Control_Box_Shadow::get_type(),
            [
                'name'     => 'image_box_shadow',
                'selector' => '{{WRAPPER}} .category-product-img',
            ]
        );

        $this->add_responsive_control(
            'image_width',
            [
                'label'          => esc_html__('Width', 'razox'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => '%',
                ],
                'tablet_default' => [
                    'unit' => '%',
                ],
                'mobile_default' => [
                    'unit' => '%',
                ],
                'size_units'     => ['%', 'px', 'vw'],
                'range'          => [
                    '%'  => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 1,
                        'max' => 1000,
                    ],
                    'vw' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .category-product-img'     => 'width: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .category-product-img img' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_height',
            [
                'label'          => esc_html__('Height', 'razox'),
                'type'           => Controls_Manager::SLIDER,
                'default'        => [
                    'unit' => 'px',
                ],
                'tablet_default' => [
                    'unit' => 'px',
                ],
                'mobile_default' => [
                    'unit' => 'px',
                ],
                'size_units'     => ['px', 'vh'],
                'range'          => [
                    'px' => [
                        'min' => 1,
                        'max' => 500,
                    ],
                    'vh' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                ],
                'selectors'      => [
                    '{{WRAPPER}} .category-product-img'     => 'height: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .category-product-img img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_object_fit',
            [
                'label'     => esc_html__('Object Fit', 'razox'),
                'type'      => Controls_Manager::SELECT,
                'condition' => [
                    'image_height[size]!' => '',
                ],
                'options'   => [
                    ''        => esc_html__('Default', 'razox'),
                    'fill'    => esc_html__('Fill', 'razox'),
                    'cover'   => esc_html__('Cover', 'razox'),
                    'contain' => esc_html__('Contain', 'razox'),
                ],
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} img' => 'object-fit: {{VALUE}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'image_padding',
            [
                'label'      => esc_html__('Padding', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .category-product-img' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_margin',
            [
                'label'      => esc_html__('Margin', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .category-product-img' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        // icon

        $this->start_controls_section(
            'icon_style',
            [
                'label' => esc_html__('Icon', 'razox'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'icon_size',
            [
                'label'      => esc_html__('Icon size', 'razox'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 30,
                        'max' => 500,
                    ],
                ],
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .category-product-img i'    => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .category-product-icon i'   => 'font-size: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .category-product-icon svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}}',
                ],
            ]
        );

        $this->start_controls_tabs('tab_icon');
        $this->start_controls_tab(
            'tab_icon_normal',
            [
                'label' => esc_html__('Normal', 'razox'),
            ]
        );
        $this->add_control(
            'icon_color',
            [
                'label'     => esc_html__('Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .category-product-img a i'                       => 'color: {{VALUE}};',
                    '{{WRAPPER}} .category-product-icon .category-product-link i' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'background_icon_color',
            [
                'label'     => esc_html__('Background Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .category-product-icon .category-product-link' => 'background-color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->start_controls_tab(
            'tab_icon_hover',
            [
                'label' => esc_html__('Hover', 'razox'),
            ]
        );
        $this->add_control(
            'icon_color_hover',
            [
                'label'     => esc_html__('Hover Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover .category-product-img a i'                       => 'color: {{VALUE}};',
                    '{{WRAPPER}} .product-cat:hover .category-product-icon .category-product-link i' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'title_style',
            [
                'label' => esc_html__('Title', 'razox'),
                'tab'   => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'tilte_typography',
                'selector' => '{{WRAPPER}} .category-title',
            ]
        );

        $this->add_responsive_control(
            'title_width',
            [
                'label'      => esc_html__('Max Width', 'razox'),
                'type'       => Controls_Manager::SLIDER,
                'range'      => [
                    'px' => [
                        'min' => 30,
                        'max' => 500,
                    ],
                ],
                'size_units' => ['px', 'em', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .category-title' => 'max-width: {{SIZE}}{{UNIT}}'
                ],
            ]
        );


        $this->add_responsive_control(
            'title_margin',
            [
                'label'      => esc_html__('Margin', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .title-count' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'title_padding',
            [
                'label'      => esc_html__('Padding', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .category-title' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->start_controls_tabs('tab_title');
        $this->start_controls_tab(
            'tab_title_normal',
            [
                'label' => esc_html__('Normal', 'razox'),
            ]
        );
        $this->add_control(
            'title_color',
            [
                'label'     => esc_html__('Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .category-title'   => 'color: {{VALUE}};',
                    '{{WRAPPER}} .category-title a' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->start_controls_tab(
            'tab_title_hover',
            [
                'label' => esc_html__('Hover', 'razox'),
            ]
        );
        $this->add_control(
            'title_color_hover',
            [
                'label'     => esc_html__('Hover Color', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .product-cat:hover .category-title' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_tab();
        $this->end_controls_tabs();

        $this->end_controls_section();

        $this->start_controls_section(
            'total_style',
            [
                'label'     => esc_html__('Total', 'razox'),
                'tab'       => Controls_Manager::TAB_STYLE,
                'condition' => ['show_count' => 'yes'],
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'     => 'total_typography',
                'selector' => '{{WRAPPER}} .category-count-wrapper',
            ]
        );

        $this->add_group_control(
            Group_Control_Border::get_type(),
            [
                'name'      => 'total_border',
                'selector'  => '{{WRAPPER}} .category-count-wrapper',
                'separator' => 'before',
            ]
        );

        $this->add_control(
            'total_border_radius',
            [
                'label'      => esc_html__('Border Radius', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors'  => [
                    '{{WRAPPER}} .category-count-wrapper' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'total_margin',
            [
                'label'      => esc_html__('Margin', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .category-count-wrapper' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'total_padding',
            [
                'label'      => esc_html__('Padding', 'razox'),
                'type'       => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', 'em'],
                'selectors'  => [
                    '{{WRAPPER}} .category-count-wrapper' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'total_color_text',
            [
                'label'     => esc_html__('Color Count Text', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .category-count-text' => 'color: {{VALUE}};',
                ],
            ]
        );
        $this->add_control(
            'total_color',
            [
                'label'     => esc_html__('Color Count', 'razox'),
                'type'      => Controls_Manager::COLOR,
                'default'   => '',
                'selectors' => [
                    '{{WRAPPER}} .category-count' => 'color: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();
        $this->get_controls_column();
        // Carousel options
        $this->get_control_carousel();
    }

    protected function get_product_categories() {
        $categories = get_terms(array(
                'taxonomy'   => 'product_cat',
                'hide_empty' => false,
            )
        );
        $results    = array();
        if (!is_wp_error($categories)) {
            foreach ($categories as $category) {
                $results[$category->slug] = $category->name;
            }
        }
        return $results;
    }

    /**
     * Render tabs widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since  1.0.0
     * @access protected
     */
    protected function render() {
        $settings = $this->get_settings_for_display();
        if (!empty($settings['categories_list']) && is_array($settings['categories_list'])) {
            $this->get_data_elementor_columns();

            $this->add_render_attribute('wrapper', 'class', 'elementor-categories-item-wrapper');
            // Item
            $this->add_render_attribute('item', 'class', 'elementor-categories-item');
            $migration_allowed = Icons_Manager::is_migration_allowed();
            ?>
            <div <?php $this->print_render_attribute_string('wrapper'); ?>>
                <div <?php $this->print_render_attribute_string('container'); ?>>
                    <div <?php $this->print_render_attribute_string('inner'); ?>>
                        <?php foreach ($settings['categories_list'] as $index => $item): ?>
                            <?php
                            //category_icon
                            $migrated              = isset($item['__fa4_migrated']['selected_icon']);
                            $is_new                = !isset($item['icon']) && $migration_allowed;
                            $class_item            = 'elementor-repeater-item-' . $item['_id'];
                            $tab_title_setting_key = $this->get_repeater_setting_key('tab_title', 'tabs', $index);
                            $this->add_render_attribute($tab_title_setting_key, ['class' => ['product-cat', $class_item],]);

                            ?>
                            <div <?php $this->print_render_attribute_string('item'); ?>>
                                <?php if (empty($item['categories'])) {
                                    echo esc_html__('Choose Category', 'razox');
                                    return;
                                }
                                $category = get_term_by('slug', $item['categories'], 'product_cat');
                                if ($item['display_background'] == 'yes') {
                                    $background = Group_Control_Image_Size::get_attachment_image_src($item['category_background']['id'], 'image', $settings);
                                }
                                if (!is_wp_error($category) && !empty($category)) {
                                    if (!empty($item['category_image']['id'])) {
                                        $image = Group_Control_Image_Size::get_attachment_image_src($item['category_image']['id'], 'image', $settings);
                                    } else {
                                        $thumbnail_id = get_term_meta($category->term_id, 'thumbnail_id', true);
                                        if (!empty($thumbnail_id)) {
                                            $image = wp_get_attachment_url($thumbnail_id);
                                        } else {
                                            $image = wc_placeholder_img_src();
                                        }
                                    }
                                    $categories_name = empty($item['categories_name']) ? esc_html($category->name) : sprintf('%s', $item['categories_name']);
                                    ?>

                                    <div <?php $this->print_render_attribute_string($tab_title_setting_key); ?>>

                                        <?php
                                        if ($settings['style'] == '1') {
                                            ?>

                                            <?php if ($item['display_background'] == 'yes') { ?>
                                                <div class="category-product-back">
                                                    <img src="<?php echo esc_url_raw($background); ?>" alt="<?php echo esc_attr($category->name); ?>">
                                                </div>
                                            <?php } ?>
                                            <div class="category-product-img">
                                                <?php if ($item['cat-select'] == 'image'): ?>
                                                    <a class="category-product-link" href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                                        <img src="<?php echo esc_url_raw($image); ?>"
                                                             alt="<?php echo esc_attr($category->name); ?>">
                                                    </a>
                                                <?php endif; ?>
                                            </div>

                                            <div class="category-product-caption">
                                                <?php if ($settings['show_count']): ?>
                                                    <div class="category-count-wrapper">
                                                        <span class="category-count"><?php echo esc_html($category->count); ?></span>
                                                        <span class="category-count-text"><?php echo esc_html__('Products', 'razox') ?></span>
                                                    </div>
                                                <?php endif; ?>

                                                <h3 class="category-title">
                                                    <a class="category-product-link" href="<?php echo esc_url(get_term_link($category)); ?>"
                                                       title="<?php echo esc_attr($category->name); ?>">
                                                        <?php echo esc_html($categories_name) ?>
                                                    </a>
                                                </h3>
                                            </div>

                                            <?php if ($settings['button_text'] && $settings['button_text'] != ''): ?>
                                                <div class="category-product-button-wrapper">
                                                    <a class="category-product-button more-link" href="<?php echo esc_url(get_term_link($category)); ?>"
                                                       title="<?php echo esc_attr($category->name); ?>">
                                                        <span class="button-text"><?php $this->print_unescaped_setting('button_text'); ?></span>
                                                    </a>
                                                </div>
                                            <?php endif;
                                            if ($is_new || $migrated) {
                                                ?>
                                                <div class="category-product-icon">
                                                    <?php
                                                    Icons_Manager::render_icon($item['selected_icon'], ['aria-hidden' => 'true']);
                                                    //                                                    Icons_Manager::render_icon($settings['categoties_icon'], ['aria-hidden' => 'true']);
                                                    ?>
                                                </div>
                                                <?php
                                            }
                                        } elseif ($settings['style'] == '2') {
                                            ?>
                                            <div class="category-product-img">
                                                <?php if ($item['display_background'] == 'yes') { ?>
                                                    <div class="category-product-back">
                                                        <img src="<?php echo esc_url_raw($background); ?>" alt="<?php echo esc_attr($category->name); ?>">
                                                    </div>
                                                <?php } ?>

                                                <a class="category-product-link" href="<?php echo esc_url(get_term_link($category)); ?>"
                                                   title="<?php echo esc_attr($category->name); ?>">
                                                    <?php if ($item['cat-select'] == 'image') { ?>
                                                        <img src="<?php echo esc_url_raw($image); ?>" alt="<?php echo esc_attr($category->name); ?>">

                                                        <?php
                                                    } else {
                                                        if ($is_new || $migrated) { ?>
                                                            <?php Icons_Manager::render_icon($item['selected_icon'], ['aria-hidden' => 'true']); ?>
                                                            <?php
                                                        }
                                                    }
                                                    ?>
                                                </a>

                                                <h3 class="category-title">
                                                    <a href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                                        <?php echo esc_html($categories_name) ?>
                                                    </a>
                                                </h3>
                                            </div>

                                            <?php if ($settings['show_count']): ?>
                                                <div class="category-product-caption">
                                                    <a class="category-count-wrapper " href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                                        <span class="category-count"><?php echo esc_html($category->count); ?></span>
                                                        <span class="category-count-text"><?php echo esc_html__('products', 'razox') ?></span>
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                            <?php
                                        } elseif ($settings['style'] == '3') {
                                            ?>

                                            <?php if ($item['display_background'] == 'yes') { ?>
                                                <div class="category-product-back">
                                                    <img src="<?php echo esc_url_raw($background); ?>" alt="<?php echo esc_attr($category->name); ?>">
                                                </div>
                                            <?php } ?>

                                            <?php if ($item['cat-select'] == 'image') { ?>
                                                <div class="category-product-img">
                                                    <a class="category-product-link" href="<?php echo esc_url(get_term_link($category)); ?>"
                                                       title="<?php echo esc_attr($category->name); ?>">
                                                        <img src="<?php echo esc_url_raw($image); ?>" alt="<?php echo esc_attr($category->name); ?>">
                                                    </a>
                                                </div>
                                                <?php
                                            } else {
                                                if ($is_new || $migrated) { ?>
                                                    <div class="category-product-icon">
                                                        <a class="category-product-link" href="<?php echo esc_url(get_term_link($category)); ?>"
                                                           title="<?php echo esc_attr($category->name); ?>">
                                                            <?php Icons_Manager::render_icon($item['selected_icon'], ['aria-hidden' => 'true']); ?>
                                                        </a>
                                                    </div>
                                                    <?php
                                                }
                                            }
                                            ?>
                                            <h3 class="category-title">
                                                <a href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                                    <?php echo esc_html($categories_name) ?>
                                                </a>
                                            </h3>
                                            <div class="category-product-caption">
                                                <?php if ($settings['show_count']): ?>
                                                    <a class="category-count-wrapper" href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                                        <span class="category-count"><?php echo esc_html($category->count); ?></span>
                                                        <span class="category-count-text"><?php echo esc_html__('products', 'razox') ?></span>
                                                    </a>
                                                <?php endif; ?>
                                            </div>
                                            <?php
                                        } elseif ($settings['style'] == '4') {
                                            ?>
                                            <?php if ($item['display_background'] == 'yes') { ?>
                                                <div class="category-product-back">

                                                    <div class="back" style="background-image: url('<?php echo esc_url_raw($background); ?>');">
                                                    </div>
                                                </div>
                                            <?php } ?>
                                            <div class="category-product-img">
                                                <a class="category-product-link" href="<?php echo esc_url(get_term_link($category)); ?>"
                                                   title="<?php echo esc_attr($category->name); ?>">
                                                    <?php if ($item['cat-select'] == 'image') { ?>
                                                        <img src="<?php echo esc_url_raw($image); ?>" alt="<?php echo esc_attr($category->name); ?>">

                                                        <?php
                                                    } else {
                                                        if ($is_new || $migrated) { ?>
                                                            <?php Icons_Manager::render_icon($item['selected_icon'], ['aria-hidden' => 'true']); ?>
                                                            <?php
                                                        }
                                                    }
                                                    ?>
                                                </a>

                                                <h3 class="category-title">
                                                    <a href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                                        <?php echo esc_html($categories_name) ?>
                                                    </a>
                                                </h3>
                                            </div>

                                            <?php if ($settings['show_count']): ?>
                                                <div class="category-product-caption">
                                                    <a class="category-count-wrapper elementor-button elementor-button-link" href="<?php echo esc_url(get_term_link($category)); ?>" title="<?php echo esc_attr($category->name); ?>">
                                                        <span class="category-count"><?php echo esc_html($category->count); ?></span>
                                                        <span class="category-count-text"><?php echo esc_html__('products', 'razox') ?></span>
                                                    </a>
                                                </div>
                                            <?php endif;
                                        }
                                        ?>
                                    </div>
                                <?php } ?>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    <?php $this->get_swiper_navigation(count($settings['categories_list'])); ?>
                </div>
            </div>

            <?php
        }
    }
}

$widgets_manager->register(new Razox_Elementor_Products_Categories());

